/**
 * @file scrollview.h
 * @brief Scroll view container component
 * 
 * Provides scrollable container functionality for content.
 */

#pragma once

#include "view.h"
#include "vertical_stack.h"
#include "horizontal_stack.h"

namespace zui
{

	/**
	 * @class ScrollView
	 * @ingroup controls_grouping
	 * @brief A view representing a ScrollView.
		 *
	 * A Scroll View provides a scrollable area for content that exceeds the visible display area.
	 * It allows users to view and interact with content by scrolling vertically or horizontally,
	 * making it suitable for displaying longer or larger content within a limited space.
	 *
	 */
	class ZUI_API ScrollView : public View
	{
	public:
		template <typename ViewT,
			typename = typename std::enable_if<
			std::is_convertible<typename std::decay<ViewT>::type, VerticalStack>::value ||
			std::is_convertible<typename std::decay<ViewT>::type, HorizontalStack>::value>::type>

		ScrollView(ViewT&& view)
			: View(View::Type::ScrollView, [this]() {
        return (this)->BuildScrollView();
      })
		{
			AddChildView(std::forward<ViewT>(view).GetViewNode());
		}

		virtual ~ScrollView() = default;

		ScrollView& horizontal(bool enabled);
		ScrollView& vertical(bool enabled);

		std::shared_ptr<ViewNode> BuildScrollView();

	private:
		void AddChildView(std::shared_ptr<ViewNode> viewNode);
	};

namespace p
{

  class ZUI_API scrollViewHorizontal : public PipeModifier
  {
  public:
    explicit scrollViewHorizontal(bool enabled);
    ~scrollViewHorizontal();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API scrollViewVertical : public PipeModifier
  {
  public:
    explicit scrollViewVertical(bool enabled);
    ~scrollViewVertical();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  namespace discover {
    namespace by_view {
      namespace scrollview {
        using zui::p::scrollViewHorizontal;
        using zui::p::scrollViewVertical;
      }
    }
  }

}

}
