/**
 * @file text.h
 * @brief Text component for displaying styled text content
 * 
 * Provides text display functionality with Prism typography support.
 */

#pragma once

#include "view.h"

namespace zui
{
  /**
   * @class Text
   * @ingroup controls_typography
   * @brief A view component for displaying styled text content.
   *
   * The Text component is the primary way to display text in zUI applications.
   * It follows Zoom's Prism design system typography guidelines and provides
   * convenient methods for common text styling patterns.
   *
   * **Key Features:**
   * - Prism typography styles (display, body, subtitle, caption, etc.)
   * - Reactive text content using @ref zui::Bind "Bind<std::string>"
   * - Flexible text wrapping and truncation
   * - Click event support for interactive text
   * - Seamless color integration with Prism color tokens
   *
   * **Common Usage Patterns:**
   *
   * @code{.cpp}
   * // Page titles and headers
   * Text("Settings")
   *     .display1()
   *     .color(textPrimary);
   *
   * // Body content with automatic wrapping
   * Text("This is the main content of your view that may span multiple lines")
   *     .body()
   *     .wrap()
   *     .color(textPrimary);
   *
   * // Secondary information
   * Text("Last updated 2 minutes ago")
   *     .caption()
   *     .color(textSecondary);
   *
   * // Interactive text (clickable)
   * Text("Learn more")
   *     .body()
   *     .color(colorPrimary)
   *     .onClick([](){
   *         // handle click
   *     });
   *
   * // Reactive content
   * State<std::string> username = "John";
   * Text(username.bind([](const std::string& name) {
   *     return "Hello, " + name + "!";
   * }))
   *     .body();
   * @endcode
   *
   * @see zui::Bind for data binding
   */
  class ZUI_API Text : public View
  {
  public:
    /**
     * @enum Typography
     * @brief Defines the typography styles available for text
     * @todo Move the enum out of the class for better reusability
     */
    enum class Typography
    {
      kDisplay1,      ///< Largest display text style
      kDisplay2,      ///< Medium display text style
      kDisplay3,      ///< Smallest display text style
      kBodyLargeBold, ///< Large body text with bold weight
      kBodyLarge,     ///< Large body text
      kBodyBold,      ///< Regular body text with bold weight
      kBody,          ///< Regular body text
      kSubtitleBold,  ///< Subtitle text with bold weight
      kSubtitle,      ///< Subtitle text
      kCaption,       ///< Small caption text
      kLabel          ///< Label text style
    };

    /**
     * @enum WrapMode
     * @brief Defines how text wrapping should behave
     */
    enum class WrapMode
    {
      kNoWrap,        ///< Text does not wrap and may overflow
      kWordWrap,      ///< Text wraps at word boundaries
      kWrapAnyWhere   ///< Text can wrap anywhere, including within words
    };

    /**
     * @brief Constructs a Text object with the given text content
     * @param text The text content to display
     * @todo Remove noNeedAddParent parameter if it's no longer needed
     */
    explicit Text(const Bind<std::string>& text);

    /// @brief Virtual destructor
    virtual ~Text() = default;
    
    std::shared_ptr<ViewNode> BuildText();
    /**
     * @brief Sets the typography style of the text
     * @param typography The typography style to apply
     * @return Reference to this text for method chaining
     */
    Text& typography(const Bind<Text::Typography>& typography);

    /**
     * @brief Sets the text color using a string color value
     * @param color The color value (e.g., "#FF0000", "red")
     * @return Reference to this text for method chaining
     */
    Text& color(const Bind<std::string>& color);

    /**
     * @brief Sets the text color using a C-string color value
     * @param color The color value as a C-string
     * @return Reference to this text for method chaining
     */
    Text& color(const char* color);

    /**
     * @brief Sets the text wrapping mode
     * @param mode The wrap mode to apply
     * @return Reference to this text for method chaining
     */
    Text& wrapMode(const Bind<WrapMode>& mode);

    /**
     * @brief Sets the callback function for click events
     * @param callback The function to call when the text is clicked
     * @return Reference to this text for method chaining
     */
    Text& onClick(std::function<void()> callback);

    /**
     * @brief Sets the text align center when word wrapping
     * @return Reference to this text for method chaining
     */
    Text& multilineCenterAlign();

    // Typography convenience methods
    
    /**
     * @brief Sets the text to display1 typography style
     * @return Reference to this text for method chaining
     */
    Text& display1();
    
    /**
     * @brief Sets the text to display2 typography style
     * @return Reference to this text for method chaining
     */
    Text& display2();
    
    /**
     * @brief Sets the text to display3 typography style
     * @return Reference to this text for method chaining
     */
    Text& display3();
    
    /**
     * @brief Sets the text to bodyLargeBold typography style
     * @return Reference to this text for method chaining
     */
    Text& bodyLargeBold();
    
    /**
     * @brief Sets the text to bodyLarge typography style
     * @return Reference to this text for method chaining
     */
    Text& bodyLarge();
    
    /**
     * @brief Sets the text to bodyBold typography style
     * @return Reference to this text for method chaining
     */
    Text& bodyBold();
    
    /**
     * @brief Sets the text to body typography style
     * @return Reference to this text for method chaining
     */
    Text& body();
    
    /**
     * @brief Sets the text to subtitleBold typography style
     * @return Reference to this text for method chaining
     */
    Text& subtitleBold();
    
    /**
     * @brief Sets the text to subtitle typography style
     * @return Reference to this text for method chaining
     */
    Text& subtitle();
    
    /**
     * @brief Sets the text to caption typography style
     * @return Reference to this text for method chaining
     */
    Text& caption();
    
    /**
     * @brief Sets the text to label typography style
     * @return Reference to this text for method chaining
     */
    Text& label();
    
    // Wrap mode convenience methods
    
    /**
     * @brief Enables text wrapping at word boundaries
     * @return Reference to this text for method chaining
     */
    Text& wrap();
    
    /**
     * @brief Disables text wrapping
     * @return Reference to this text for method chaining
     */
    Text& noWrap();

    bool operator==(const Text& other) const;

protected:
    Text(const Bind<std::string>& text, View::Type viewType, std::function<std::shared_ptr<ViewNode>()> builder);
};

namespace p
{

  // Text-specific pipe modifiers
  class ZUI_API textTypography : public PipeModifier
  {
  public:
    explicit textTypography(const Bind<Text::Typography>& typography);
    ~textTypography();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textColor : public PipeModifier
  {
  public:
    explicit textColor(const Bind<std::string>& color);
    explicit textColor(const char* color);
    ~textColor();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textWrapMode : public PipeModifier
  {
  public:
    explicit textWrapMode(const Bind<Text::WrapMode>& mode);
    ~textWrapMode();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textMultilineCenterAlign : public PipeModifier
  {
  public:
    textMultilineCenterAlign();
    ~textMultilineCenterAlign();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  // Typography convenience modifiers
  class ZUI_API textDisplay1 : public PipeModifier
  {
  public:
    textDisplay1();
    ~textDisplay1();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textDisplay2 : public PipeModifier
  {
  public:
    textDisplay2();
    ~textDisplay2();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textDisplay3 : public PipeModifier
  {
  public:
    textDisplay3();
    ~textDisplay3();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textBodyLargeBold : public PipeModifier
  {
  public:
    textBodyLargeBold();
    ~textBodyLargeBold();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textBodyLarge : public PipeModifier
  {
  public:
    textBodyLarge();
    ~textBodyLarge();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textBodyBold : public PipeModifier
  {
  public:
    textBodyBold();
    ~textBodyBold();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textBody : public PipeModifier
  {
  public:
    textBody();
    ~textBody();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textSubtitleBold : public PipeModifier
  {
  public:
    textSubtitleBold();
    ~textSubtitleBold();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textSubtitle : public PipeModifier
  {
  public:
    textSubtitle();
    ~textSubtitle();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textCaption : public PipeModifier
  {
  public:
    textCaption();
    ~textCaption();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textNoWrap : public PipeModifier
  {
  public:
    textNoWrap();
    ~textNoWrap();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textWrap : public PipeModifier
  {
  public:
    textWrap();
    ~textWrap();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API textLabel : public PipeModifier
  {
  public:
    textLabel();
    ~textLabel();
  protected:
    void apply(View& v) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  namespace discover {
    namespace by_view {
      namespace text {
        using zui::p::textTypography;
        using zui::p::textColor;
        using zui::p::textWrapMode;
        using zui::p::textMultilineCenterAlign;
        using zui::p::textDisplay1;
        using zui::p::textDisplay2;
        using zui::p::textDisplay3;
        using zui::p::textBodyLargeBold;
        using zui::p::textBodyLarge;
        using zui::p::textBodyBold;
        using zui::p::textBody;
        using zui::p::textSubtitleBold;
        using zui::p::textSubtitle;
        using zui::p::textCaption;
        using zui::p::textNoWrap;
        using zui::p::textWrap;
        using zui::p::textLabel;
      }
    }
  }

} // namespace p

} // namespace zui
