/**
 * @file ZoomSDKSettingVideoEffectController.h
 * @brief Interface declarations related to video effect settings in Zoom SDK.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <ZoomSDK/ZoomSDKErrors.h>

@class ZoomSDKVideoLipSyncAvatarPreviewHelper;
@class ZoomSDKSettingTestVideoDeviceHelper;

NS_ASSUME_NONNULL_BEGIN

/**
 * @class ZoomSDK3DAvatarImageInfo
 * @brief Represents information of a 3D avatar image.
 */
@interface ZoomSDK3DAvatarImageInfo: NSObject

/**
 * @brief Determines whether the current image is being used.
 */
@property(nonatomic,assign,readonly)BOOL isSelected;

/**
 * @brief Determines whether it is the most recently used image.
 */
@property(nonatomic,assign,readonly)BOOL isLastUsed;

/**
 * @brief Gets the current image's file path.
 */
@property(nonatomic,copy,readonly,nullable)NSString* imageFilePath;

/**
 * @brief Gets the current image's name.
 */
@property(nonatomic,copy,readonly,nullable)NSString* imageName;

/**
 * @brief Gets the current image's index.
 */
@property(nonatomic,assign,readonly)int index;
@end

/**
 * @class ZoomSDKCustom3DAvatarElementImageInfo
 * @brief Represents information of a custom 3D avatar element image.
 */
@interface ZoomSDKCustom3DAvatarElementImageInfo : NSObject

/**
 * @brief Gets the current element image's type.
 */
@property(nonatomic, assign, readonly)ZoomSDKCustom3DAvatarElementImageType custom3DAvatarElementImageType;

/**
 * @brief Determines whether the current element image is being used.
 */
@property(nonatomic, assign, readonly)BOOL isSelected;

/**
 * @brief Gets the current element image's file path.
 */
@property(nonatomic, copy, readonly, nullable)NSString* imageFilePath;

/**
 * @brief Gets the current element image's name.
 */
@property(nonatomic, copy, readonly, nullable)NSString* imageName;

@end

/**
 * @class ZoomSDKCustom3DAvatarElementColorInfo
 * @brief Custom 3D avatar element color information interface.
 * @note These interfaces are only valid for the custom UI mode.
 */
@interface ZoomSDKCustom3DAvatarElementColorInfo : NSObject

/**
 * @brief Gets the current element color's type.
 */
@property(nonatomic, assign, readonly)ZoomSDKCustom3DAvatarElementColorType custom3DAvatarElementColorType;

/**
 * @brief Gets the current element's color in RGB format.
 */
@property(nonatomic, retain, readonly)NSColor* color;

/**
 * @brief Determines whether the current element color is being used.
 */
@property(nonatomic, assign, readonly)BOOL isSelected;

/**
 * @brief Gets the current element color's name.
 */
@property(nonatomic, copy, readonly, nullable)NSString* imageName;
@end


/**
 * @protocol ZoomSDKCustom3DAvatarElementSettingContextDelegate
 * @brief Custom 3D avatar element setting context callback event.
 */
@protocol ZoomSDKCustom3DAvatarElementSettingContextDelegate <NSObject>

/**
 * @brief Callback event when the custom 3D avatar element image model data download completes.
 * @param isSuccess YES if the custom 3D avatar element image model data has been downloaded successfully, NO otherwise.
 * @param imageInfo The image info for the downloaded custom 3D avatar element image.
 */
- (void)onCustom3DAvatarElementImageModelDataDownloaded:(BOOL)isSuccess elementImage:(ZoomSDKCustom3DAvatarElementImageInfo*)imageInfo;
@end

/**
 * @class ZoomSDKCustom3DAvatarElementSettingContext
 * @brief Context interface for configuring custom 3D avatar elements during avatar creation or editing.
 *
 * These interfaces are provided after calling
 * \link startCreateCustom3DAvatar: \endlink or \link startEditCustom3DAvatar: avatarImage: \endlink, and is used to configure
 * the visual elements of a custom 3D avatar, such as:
 * - Selecting a specific avatar element image (model)
 * - Downloading and checking readiness of element model data
 * - Applying a color to the selected avatar element
 *
 * The context represents an active avatar creation or editing session.
 * All operations performed through this context affect the
 * custom 3D avatar currently being created or edited.
 *
 * @note
 * - These interfaces are only valid during the custom 3D avatar creation or editing process.
 * - Set delegate first before using any other method.
 * - Do not delete the returned image of color lists. They are managed by the SDK.
 * - Model data must be downloaded and ready before applying an image.
 * - These interfaces are only valid for the custom UI mode.
 */
@interface ZoomSDKCustom3DAvatarElementSettingContext : NSObject

/**
 * @brief Gets or sets the delegate for receiving custom 3D avatar element setting events.
 */
@property(nonatomic, assign, nullable)id<ZoomSDKCustom3DAvatarElementSettingContextDelegate> delegate;

/**
 * @brief Gets the list of available custom 3D avatar element images.
 * @return If there are images in the list, the return value is an NSArray of custom 3D avatar element image info objects. Otherwise, this function fails and returns nil.
 */
- (NSArray<ZoomSDKCustom3DAvatarElementImageInfo*>*_Nullable)getCustom3DAvatarElementImageList;

/**
 * @brief Checks whether the model data for a specific custom 3D avatar element image has been fully downloaded and is ready for use.
 * Before an avatar element image can be applied, its model data must be fully downloaded and ready.
 * @param imageInfo The avatar element image info to check.
 * @return YES if the model data is ready. Otherwise, NO.
 */
- (BOOL)isCustom3DAvatarElementImageModelDataReady:(ZoomSDKCustom3DAvatarElementImageInfo*)imageInfo;

/**
 * @brief Downloads the model data required for a specific custom 3D avatar element image.
 * This should be called if \link isCustom3DAvatarElementImageModelDataReady \endlink  returns NO for the specified image.
 * @param imageInfo The avatar element image whose model data should be downloaded.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note This should be called before using the image if its model data is not yet ready.
 */
- (ZoomSDKError)downloadCustom3dAvatarElementImageModelData:(ZoomSDKCustom3DAvatarElementImageInfo*)imageInfo;

/**
 * @brief Applies a custom 3D avatar element image to the avatar being created or edited.
 * @param imageInfo The avatar element image to apply.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note The model data for the image must be ready before calling this method.
 */
- (ZoomSDKError)setCustom3DAvatarElementImage:(ZoomSDKCustom3DAvatarElementImageInfo*)imageInfo;

/**
 * @brief Gets the list of the custom 3D avatar element colors.
 * @return If there are colors in the list, the return value is an NSArray of custom 3D avatar element color info objects. Otherwise, this function fails and returns nil.
 */
- (NSArray<ZoomSDKCustom3DAvatarElementColorInfo*>*_Nullable)getCustom3DAvatarElementColorList;

/**
 * @brief Applies a color to the avatar being created or edited.
 * @param colorInfo The color information to apply.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setCustom3DAvatarElementColor:(ZoomSDKCustom3DAvatarElementColorInfo*)colorInfo;
@end

/**
 * @protocol ZoomSDK3DAvatarSettingDelegate
 * @brief 3D avatar context callback event.
 */
@protocol ZoomSDK3DAvatarSettingDelegate <NSObject>

/**
 * @brief Callback event when all 3D avatar items' thumbnails have been downloaded.
 */
- (void)on3DAvatarItemThumbnailsDownloaded;

/**
 * @brief Callback event when the selected 3D avatar item is downloading.
 * @param index The index of the selected 3D avatar item.
 */
- (void)on3DAvatarItemDataDownloading:(int)index;

/**
 * @brief Callback event when the selected 3D avatar item download completes.
 * @param isSuccess YES if the selected 3D avatar item has been downloaded successfully, NO otherwise.
 * @param index The index of the selected 3D avatar item.
 */
- (void)on3DAvatarItemDataDownloaded:(BOOL)isSuccess index:(int)index;

/**
 * @brief Callback event when the custom 3D avatar image model data download completes.
 * @param isSuccess YES if the custom 3D avatar image model data has been downloaded successfully, NO otherwise.
 * @param image The custom 3D avatar image info that was downloaded.
 */
- (void)onCustom3DAvatarImageModelDataDownloaded:(BOOL)isSuccess avatarImage:(ZoomSDK3DAvatarImageInfo *)image;

/**
 * @brief Callback event when the custom 3D avatar default image model data download completes.
 * @param isSuccess YES if the custom 3D avatar default image model data has been downloaded successfully, NO otherwise.
 */
- (void)onCustom3DAvatarDefaultImageModelDataDownloaded:(BOOL)isSuccess;
@end

/**
 * @class ZoomSDK3DAvatarSetting
 * @brief 3D avatar setting interface.
 */
@interface ZoomSDK3DAvatarSetting : NSObject
{
    ZoomSDKVideoLipSyncAvatarPreviewHelper* lipsyncAvatarPreviewHelper;
    id<ZoomSDK3DAvatarSettingDelegate>       _delegate;
}
/**
 * @brief Gets or sets the delegate for receiving 3D avatar setting events.
 */
@property(nonatomic,assign, nullable)id<ZoomSDK3DAvatarSettingDelegate> delegate;

/**
 * @brief Determines whether the 3D avatar feature is supported by the video device.
 * @return YES if the video device supports the 3D avatar feature. Otherwise, NO.
 */
- (BOOL)is3DAvatarSupportedByDevice;

/**
 * @brief Determines whether the 3D avatar feature is enabled.
 * @return YES if the 3D avatar feature is enabled. Otherwise, NO.
 */
- (BOOL)is3DAvatarEnabled;

/**
 * @brief Gets the list of the 3D avatar images.
 * @return If the function succeeds, it returns an NSArray of 3D avatar image info objects. Otherwise, this function fails and returns nil.
 */
- (NSArray<ZoomSDK3DAvatarImageInfo*>*_Nullable)get3DAvatarImageList;

/**
 * @brief Specifies an image to be the 3D avatar image.
 * @param image The image to use.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)set3DAvatarImage:(ZoomSDK3DAvatarImageInfo*)image;

/**
 * @brief Gets the object to test the video device.
 * @return If the function succeeds, it returns a ZoomSDKSettingTestVideoDeviceHelper object. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKSettingTestVideoDeviceHelper*_Nullable)getTestVideoDeviceHelper;

/**
 * @brief Gets the object to preview the lip-sync avatar.
 * @return If the function succeeds, it returns a ZoomSDKVideoLipSyncAvatarPreviewHelper object. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKVideoLipSyncAvatarPreviewHelper*_Nullable)getLipSyncAvatarPreviewHelper;

/**
 * @brief Enables or disables the 3D avatar effect when joining a meeting.
 * @param enable YES to enable the 3D avatar effect, NO to disable.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enable3DAvatarEffectForAllMeeting:(BOOL)enable;

/**
 * @brief Determines whether the 3D avatar is enabled when joining a meeting.
 * @return YES if the 3D avatar is enabled when joining a meeting. Otherwise, NO.
 */
- (BOOL)is3DAvatarEffectForAllMeetingEnabled;

/**
 * @brief Determines whether the custom 3D avatar feature is enabled.
 * @return YES if the custom 3D avatar feature is enabled. Otherwise, NO.
 */
- (BOOL)isCustom3DAvatarEnabled;

/**
 * @brief Gets the list of available custom 3D avatar images.
 * Each image represents a custom 3D avatar that can be selected, edited, duplicated, or deleted.
 * @return If there are images in the list, the return value is an NSArray of custom 3D avatar image info objects. Otherwise, this function fails and returns nil.
 * @note This interface is only valid for the custom UI mode.
 */
- (NSArray<ZoomSDK3DAvatarImageInfo*>*_Nullable)getCustom3DAvatarImageList;

/**
 * @brief Checks whether the model data for a specific custom 3D avatar image is ready.
 * The avatar image can only be used after its model data has been fully downloaded and prepared.
 * @param imageInfo The custom 3D avatar image info to check.
 * @return YES if the model data for the specified image has been downloaded and is ready. Otherwise, NO.
 * @note This interface is only valid for the custom UI mode.
 */
- (BOOL)isCustom3DAvatarImageModelDataReady:(ZoomSDK3DAvatarImageInfo*)imageInfo;

/**
 * @brief Downloads the model data required for a specific custom 3D avatar image.
 * Call this method if \link isCustom3DAvatarImageModelDataReady \endlink returns NO for the specified custom 3D avatar image.
 * @param imageInfo The custom 3D avatar image whose model data should be downloaded.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note This should be called before calling \link setCustom3DAvatarImage: \endlink if its model data is not yet ready.
 * This interface is only valid for the custom UI mode.
 */
- (ZoomSDKError)downloadCustom3DAvatarImageModelData:(ZoomSDK3DAvatarImageInfo*)imageInfo;

/**
 * @brief Applies a custom 3D avatar image as the active avatar.
 * @param imageInfo The custom 3D avatar image to apply.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note The model data for the avatar image must be fully downloaded and ready before calling this method.
 * This interface is only valid for the custom UI mode.
 */
- (ZoomSDKError)setCustom3DAvatarImage:(ZoomSDK3DAvatarImageInfo*)imageInfo;

/**
 * @brief Checks whether the model data for default custom 3D avatar elements image have been fully downloaded and are ready for use.
 * This method is typically used before starting the custom 3D avatar creation process to ensure default elements data is available.
 * @return YES if the model data for all custom 3D avatar elements image have been downloaded and are ready. Otherwise, NO.
 * @note This interface is only valid for the custom UI mode.
 */
- (BOOL)isCustom3DAvatarDefaultImageModelDataReady;

/**
 * @brief Downloads the model data required for a default custom 3D avatar image.
 * Call this method if  \link isCustom3DAvatarDefaultImageModelDataReady \endlink returns NO for the default custom 3D avatar image.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note This should be called before calling \link startCreateCustom3DAvatar: \endlink if the default image model data is not yet ready.
 * This interface is only valid for the custom UI mode.
 */
- (ZoomSDKError)downloadCustom3DAvatarDefaultImageModelData;

/**
 * @brief Starts creating a new custom 3D avatar.
 * This method initializes a custom 3D avatar creation session
 * and returns a \link  ZoomSDKCustom3DAvatarElementSettingContext \endlink instance for configuring avatar elements (image, model data, color, etc.).
 * @param previewView The view used to render the avatar preview.
 * @return If the function succeeds, the return value is a \link  ZoomSDKCustom3DAvatarElementSettingContext \endlink object. Otherwise, this function fails and returns nil.
 * @note The model data for all elements image must be ready before calling this method. Otherwise returns nil.
 * If the function succeeds, before calling \link  finishCreateCustom3DAvatar: \endlink,
 * calling 3D-avatar-related API will result in an error. The maximum number of custom 3D avatars is 25. Exceeding this limit will result in an error.
 * This interface is only valid for the custom UI mode.
 */
- (ZoomSDKCustom3DAvatarElementSettingContext*_Nullable)startCreateCustom3DAvatar:(NSView*)previewView;

/**
 * @brief Finishes creating a custom 3D avatar.
 * This method ends the custom 3D avatar creation session that was started by \link startCreateCustom3DAvatar: \endlink.
 * @param save YES to apply the selected avatar elements and save the newly created custom 3D avatar, NO to discard all changes and cancel the creation.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note This interface is only valid for the custom UI mode.
 */
- (ZoomSDKError)finishCreateCustom3DAvatar:(BOOL)save;

/**
 * @brief Starts editing an existing custom 3D avatar.
 * This method starts an editing session for the specified custom 3D avatar
 * and returns a \link ZoomSDKCustom3DAvatarElementSettingContext \endlink instance for modifying avatar elements such as images and colors.
 * @param previewView The view used to render the avatar preview.
 * @param imageInfo The image info of the custom 3D avatar to edit.
 * @return If the function succeeds, the return value is a \link ZoomSDKCustom3DAvatarElementSettingContext \endlink object. Otherwise, this function fails and returns nil.
 * @note The model data for the avatar image must be fully downloaded and ready before calling this method.
 * After this method succeeds and before calling \link finishEditCustom3DAvatar: \endlink, invoking other 3D avatar-related APIs will result in an error.
 * This interface is only valid for the custom UI mode.
 */
- (ZoomSDKCustom3DAvatarElementSettingContext*_Nullable)startEditCustom3DAvatar:(NSView*)previewView avatarImage:(ZoomSDK3DAvatarImageInfo*)imageInfo;

/**
 * @brief Finishes editing a custom 3D avatar.
 * This method ends the custom 3D avatar editing session that was started by \link startEditCustom3DAvatar: avatarImage: \endlink.
 * @param save YES to apply the selected avatar elements and save the edited custom 3D avatar, NO to discard all changes and cancel the editing.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note This interface is only valid for the custom UI mode.
 */
- (ZoomSDKError)finishEditCustom3DAvatar:(BOOL)save;

/**
 * @brief Duplicates a custom 3D avatar.
 * @param imageInfo The custom 3D avatar image to duplicate.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note The maximum number of custom 3D avatars is 25. Exceeding this limit will result in an error.
 * This interface is only valid for the custom UI mode.
 */
- (ZoomSDKError)duplicateCustom3DAvatarImage:(ZoomSDK3DAvatarImageInfo*)imageInfo;

/**
 * @brief Deletes a custom 3D avatar.
 * @param imageInfo The custom 3D avatar image to delete.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note This interface is only valid for the custom UI mode.
 */
- (ZoomSDKError)deleteCustom3DAvatarImage:(ZoomSDK3DAvatarImageInfo*)imageInfo;
@end

NS_ASSUME_NONNULL_END

