/**
 * @file popover.h
 * @brief Popover and PopoverButton components
 * 
 * Contains overlay and navigation components for contextual information display.
 */

#pragma once

#include "view.h"
#include "presentation.h"
#include <string>
#include <functional>
#include "zui_data_define.h"

namespace zui
{
	/**
	 * @class Popover
	 * @brief A view component that displays contextual information in a popup overlay.
	 * @ingroup zui_overlay_views
	 * 
	 * The Popover component provides a floating overlay that can display rich content
	 * including title, description, links, and action buttons. It supports:
	 * - Multiple positioning directions
	 * - Title and content text
	 * - Action buttons (primary, secondary)
	 * - Links with click handling
	 * - Optional close button
	 * - Event callbacks for show/hide/click actions
	 * 
	 * @example
	 * @code
	 * // Basic informational popover
	 * Popover("This feature helps you manage your settings")
	 *   .title("Settings Helper")
	 *   .direction(Popover::Direction::kDown)
	 *   .close_button(true);
	 * 
	 * // Action popover with buttons
	 * Popover("Are you sure you want to delete this item?")
	 *   .title("Confirm Deletion")
	 *   .primary_action("Delete")
	 *   .secondary_action("Cancel")
	 *   .onClick([](Popover::ButtonType type, const std::string& text) {
	 *     if (type == Popover::ButtonType::kPrimary) {
	 *       deleteItem();
	 *     }
	 *   });
	 * @endcode
	 */
	class ZUI_API Popover : public Presentation
	{
	public:
		/**
		 * @enum Direction
		 * @brief Defines the positioning direction of the popover relative to its anchor
		 */
		enum class Direction
		{
			kDown,        ///< Popover appears below the anchor
			kDownLeft,    ///< Popover appears below and to the left of the anchor
			kDownRight,   ///< Popover appears below and to the right of the anchor
			kUp,          ///< Popover appears above the anchor
			kUpLeft,      ///< Popover appears above and to the left of the anchor
			kUpRight,     ///< Popover appears above and to the right of the anchor
			kLeft,        ///< Popover appears to the left of the anchor
			kLeftTop,     ///< Popover appears to the left and above the anchor
			kLeftBottom,  ///< Popover appears to the left and below the anchor
			kRight,       ///< Popover appears to the right of the anchor
			kRightTop,    ///< Popover appears to the right and above the anchor
			kRightBottom  ///< Popover appears to the right and below the anchor
		};

		/**
		 * @enum ButtonType
		 * @brief Defines the types of buttons available in the popover
		 */
		enum class ButtonType
		{
			kClose,     ///< Close button (X button)
			kPrimary,   ///< Primary action button
			kSecondary, ///< Secondary action button
			kLink,      ///< Link button
		};

		/**
		 * @brief Constructs a Popover with initial content
		 * @param content The main content text to display in the popover
		 * @todo Consider making explicit and clarify content parameter usage
		 */
		Popover(const zui::Bind<std::string>& content);

		/// @brief Virtual destructor
		virtual ~Popover() = default;

		/// @brief Builds the presentation node for this popover
		virtual PresentationNodePtr Build() override;

		/// @brief Gets the presentation type
		virtual Presentation::Type GetType() const override { return Presentation::Type::kPopover; }

		/**
		 * @brief Sets the title of the popover
		 * @param title The title text to display at the top of the popover
		 * @return Reference to this popover for method chaining
		 */
		Popover& title(const zui::Bind<std::string>& title);

		/**
		 * @brief Sets the content of the popover
		 * @param content The main content text to display
		 * @return Reference to this popover for method chaining
		 */
		Popover& content(const zui::Bind<std::string>& content);

		/**
		 * @brief Sets a clickable link in the popover
		 * @param text The text to display for the link
		 * @param target The target URL or identifier for the link
		 * @return Reference to this popover for method chaining
		 * @todo Consider accepting a Link component as parameter instead
		 */
		Popover& link(const zui::Bind<std::string>& text, const zui::Bind<std::string>& target);

		/**
		 * @brief Sets the primary action button text
		 * @param text The text to display on the primary action button
		 * @return Reference to this popover for method chaining
		 * @todo Consider accepting a BasicButton component as parameter instead
		 */
		Popover& primary_action(const zui::Bind<std::string>& text);

		/**
		 * @brief Sets the secondary action button text
		 * @param text The text to display on the secondary action button
		 * @return Reference to this popover for method chaining
		 * @todo Consider accepting a BasicButton component as parameter instead
		 */
		Popover& secondary_action(const zui::Bind<std::string>& text);

		/**
		 * @brief Sets the positioning direction of the popover
		 * @param direction The direction where the popover should appear
		 * @return Reference to this popover for method chaining
		 */
		Popover& direction(Popover::Direction direction);

		/**
		 * @brief Enables or disables the close button
		 * @param enabled True to show the close button, false to hide it
		 * @return Reference to this popover for method chaining
		 */
		Popover& close_button(bool enabled);

		/**
		 * @brief Sets the visibility of the popover
		 * @param visible True to show the popover, false to hide it
		 * @return Reference to this popover for method chaining
		 */
		Popover& visible(const zui::Bind<bool>& visible);

		/**
		 * @brief Set the background image of the popover.
		 * @param[in] image_path The path or URL of the background image.
		 */
		Popover& illustration(const zui::Bind<std::string>& path);

		/**
		 * @brief Set the size of the illustration.
		 * @param[in] width The width of the illustration.
		 * @param[in] height The height of the illustration.
		 */
		Popover& illustrationSize(const zui::Bind<Layout::Size>& size);

		/**
		 * @brief Sets the callback function for popover show events
		 * @param callback The function to call when the popover is shown
		 * @return Reference to this popover for method chaining
		 */
		Popover& onShow(std::function<void()> callback);

		/**
		 * @brief Sets the callback function for popover hide events
		 * @param callback The function to call when the popover is hidden
		 * @return Reference to this popover for method chaining
		 */
		Popover& onHide(std::function<void()> callback);

		/**
		 * @brief Sets the callback function for popover destroy events
		 * @param callback The function to call when the popover is destroyed
		 * @return Reference to this popover for method chaining
		 */
		Popover& onDestroy(std::function<void()> callback);

		/**
		 * @brief Sets the callback function for button click events
		 * @param callback The function to call when any button is clicked
		 * @return Reference to this popover for method chaining
		 */
		Popover& onClick(std::function<void(Popover::ButtonType, const std::string&)> callback);

		/**
		 * @brief Sets the callback function for description link activation
		 * @param callback The function to call when a link in the description is activated
		 * @return Reference to this popover for method chaining
		 */
		Popover& onDescLinkActive(std::function<void(const std::string&)> callback);

		/**
		 * @brief Sets the callback function for mouse enter events
		 * @param callback The function to call when the mouse enters the popover
		 * @return Reference to this popover for method chaining
		 */
		Popover& onMouseEnter(std::function<void()> callback);

		/**
		 * @brief Sets the callback function for mouse exit events
		 * @param callback The function to call when the mouse exits the popover
		 * @return Reference to this popover for method chaining
		 */
		Popover& onMouseExit(std::function<void()> callback);

		/**
		 * @brief Sets whether HTML rendering is enabled for the content
		 * 
		 * When enabled, the content will be parsed and rendered as HTML.
		 * 
		 * @param enabled True to enable HTML parsing, false for plain text
		 * @return Reference to this popover for method chaining
		 */
		Popover& htmlContent(const zui::Bind<bool>& enabled);
	};

	/**
	 * @class enable_popover
	 * @brief A mixin template class that adds popover functionality to components.
	 * @ingroup zui_overlay_views
	 * 
	 * This template class can be inherited by view components to provide
	 * popover attachment capabilities, allowing any component to display
	 * popovers when interacted with.
	 * 
	 * @tparam T The type of the component that will have popover functionality
	 * 
	 * @example
	 * @code
	 * // Usage in a component class
	 * class MyButton : public View, public enable_popover<MyButton>
	 * {
	 *   // Component implementation...
	 * };
	 * 
	 * // Using the popover functionality
	 * MyButton()
	 *   .popover(Popover("Help text").title("Button Help"));
	 * @endcode
	 */
	template <typename T>
	class ZUI_API enable_popover
	{
	public:
		/**
		 * @brief Attaches a popover to the component
		 * @param popover The popover to attach to this component
		 * @return Reference to the component for method chaining
		 */
		T& popover(const Popover& popover);
	};

	/**
	 * @class PopoverButton
	 * @brief A specialized button component that can display popovers.
	 * @ingroup zui_navigation_views
	 * 
	 * PopoverButton provides a pre-configured button specifically designed
	 * for showing contextual information through popovers. It supports:
	 * - Multiple visual types (info, question, alert)
	 * - Various sizes (small, medium, large)
	 * - Direct popover content or popover object attachment
	 * - Built-in popover functionality
	 * 
	 * @example
	 * @code
	 * // Info button with simple content
	 * PopoverButton("This explains the feature", 
	 *               PopoverButton::Type::kInfo, 
	 *               PopoverButton::Size::kSmall);
	 * 
	 * // Question button with custom popover
	 * PopoverButton(
	 *   Popover("Detailed help information")
	 *     .title("Help")
	 *     .primary_action("Got it"),
	 *   PopoverButton::Type::kQuestion
	 * );
	 * @endcode
	 */
	class ZUI_API PopoverButton
		: public View
		, public enable_popover<PopoverButton>
	{
	public:
		/**
		 * @enum Type
		 * @brief Defines the visual type of the popover button
		 */
		enum class Type
		{
			kInfo,     ///< Information button (typically "i" icon)
			kQuestion, ///< Question button (typically "?" icon)
			kAlert,    ///< Alert button (typically "!" icon)
		};

		/**
		 * @enum Size
		 * @brief Defines the size of the popover button
		 */
		enum class Size
		{
			kSmall,  ///< Small popover button
			kMedium, ///< Medium popover button (default)
			kLarge,  ///< Large popover button
		};

	public:
		/**
		 * @brief Constructs a PopoverButton with simple content
		 * @param content The text content to display in the popover
		 * @param type The visual type of the button
		 * @param size The size of the button
		 */
		PopoverButton(const std::string& content, Type type = Type::kInfo, Size size = Size::kMedium);

		/**
		 * @brief Constructs a PopoverButton with a custom popover
		 * @param popover The popover object to display
		 * @param type The visual type of the button
		 * @param size The size of the button
		 */
		PopoverButton(const Popover& popover, Type type = Type::kInfo, Size size = Size::kMedium);

		/// @brief Virtual destructor
		virtual ~PopoverButton() = default;

		/**
		 * @brief Sets the visual type of the popover button
		 * @param type The type to apply
		 * @return Reference to this popover button for method chaining
		 */
		PopoverButton& type(Type type);

		/**
		 * @brief Sets the size of the popover button
		 * @param size The size to apply
		 * @return Reference to this popover button for method chaining
		 */
		PopoverButton& size(Size size);
	};

namespace p
{

  // Presentation-specific pipe modifiers
  class ZUI_API popoverTitle : public PipeModifierPresentation
  {
  public:
    explicit popoverTitle(const Bind<std::string>& titleText);
    ~popoverTitle();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverVisible : public PipeModifierPresentation
  {
  public:
    explicit popoverVisible(const Bind<bool>& isVisible);
    ~popoverVisible();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  // Popover-specific pipe modifiers
  class ZUI_API popoverContent : public PipeModifierPresentation
  {
  public:
    explicit popoverContent(const Bind<std::string>& content);
    ~popoverContent();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverLink : public PipeModifierPresentation
  {
  public:
    explicit popoverLink(const Bind<std::string>& text, const Bind<std::string>& target);
    ~popoverLink();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverPrimaryAction : public PipeModifierPresentation
  {
  public:
    explicit popoverPrimaryAction(const Bind<std::string>& text);
    ~popoverPrimaryAction();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverSecondaryAction : public PipeModifierPresentation
  {
  public:
    explicit popoverSecondaryAction(const Bind<std::string>& text);
    ~popoverSecondaryAction();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverDirection : public PipeModifierPresentation
  {
  public:
    explicit popoverDirection(Popover::Direction direction);
    ~popoverDirection();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverCloseButton : public PipeModifierPresentation
  {
  public:
    explicit popoverCloseButton(bool enabled);
    ~popoverCloseButton();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverIllustration : public PipeModifierPresentation
  {
  public:
    explicit popoverIllustration(const Bind<std::string>& path);
    ~popoverIllustration();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverIllustrationSize : public PipeModifierPresentation
  {
  public:
    explicit popoverIllustrationSize(const Bind<Layout::Size>& size);
    ~popoverIllustrationSize();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverOnShow : public PipeModifierPresentation
  {
  public:
    explicit popoverOnShow(std::function<void()> callback);
    ~popoverOnShow();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverOnHide : public PipeModifierPresentation
  {
  public:
    explicit popoverOnHide(std::function<void()> callback);
    ~popoverOnHide();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverOnDestroy : public PipeModifierPresentation
  {
  public:
    explicit popoverOnDestroy(std::function<void()> callback);
    ~popoverOnDestroy();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverOnClick : public PipeModifierPresentation
  {
  public:
    explicit popoverOnClick(std::function<void(Popover::ButtonType, const std::string&)> callback);
    ~popoverOnClick();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  class ZUI_API popoverOnDescLinkActive : public PipeModifierPresentation
  {
  public:
    explicit popoverOnDescLinkActive(std::function<void(const std::string&)> callback);
    ~popoverOnDescLinkActive();
  protected:
    void apply(Presentation& p) const override;
  private:
    class Impl;
    Impl* _pImpl;
  };

  namespace discover {
    namespace by_view {
      namespace popover {
        using zui::p::popoverTitle;
        using zui::p::popoverVisible;
        using zui::p::popoverContent;
        using zui::p::popoverLink;
        using zui::p::popoverPrimaryAction;
        using zui::p::popoverSecondaryAction;
        using zui::p::popoverDirection;
        using zui::p::popoverCloseButton;
        using zui::p::popoverIllustration;
        using zui::p::popoverIllustrationSize;
        using zui::p::popoverOnShow;
        using zui::p::popoverOnHide;
        using zui::p::popoverOnDestroy;
        using zui::p::popoverOnClick;
        using zui::p::popoverOnDescLinkActive;
      }
    }
  }

}
}
