#ifndef _ZUI_TOOLTIP_H_
#define _ZUI_TOOLTIP_H_

#include "presentation.h"
#include <string>

namespace zui
{

	/**
	 * @class ToolTip
	 * @brief A view component that displays contextual information on hover.
	 * 
	 * The ToolTip component provides small informational overlays that appear
	 * when users hover over UI elements. It supports:
	 * - Simple text content
	 * - Keyboard shortcut display
	 * - Modifier key combinations
	 * - Automatic positioning
	 * 
	 * @example
	 * @code
	 * // Simple text tooltip
	 * ToolTip("Click to save your changes");
	 * 
	 * // Tooltip with keyboard shortcut
	 * ToolTip("Save file")
	 *   .keyInput("S")
	 *   .keyModifiers(ToolTip::Modifier::kCommand);
	 * 
	 * // Complex shortcut tooltip
	 * ToolTip("Copy selection")
	 *   .keyInput("C")
	 *   .keyModifiers(ToolTip::Modifier::kCommand, ToolTip::Modifier::kShift);
	 * @endcode
	 */
	class ZUI_API ToolTip : public Presentation
	{
	public:
		/**
		 * @enum Modifier
		 * @brief Defines keyboard modifier keys for shortcuts displayed in tooltips
		 */
		enum class Modifier : uint32_t
		{
			kShift,     ///< Shift key modifier
			kControl,   ///< Control key modifier (Ctrl on Windows/Linux)
			kOption,    ///< Option key modifier (Alt on Windows/Linux)
			kAlternate, ///< Alternate key modifier
			kCommand,   ///< Command key modifier (Cmd on macOS, Win key on Windows)
		};

		/**
		 * @brief Constructs a ToolTip with the specified text
		 * @param text The text content to display in the tooltip
		 */
		ToolTip(const std::string& text);

		/// @brief Virtual destructor
		virtual ~ToolTip() = default;

		/// @brief Builds the presentation node for this tooltip
		virtual PresentationNodePtr Build() override;

		/// @brief Gets the presentation type
		virtual Presentation::Type GetType() const override { return Presentation::Type::kTooltip; }

		/**
		 * @brief Sets the text content of the tooltip
		 * @param text The text to display
		 * @return Reference to this tooltip for method chaining
		 */
		ToolTip& text(const std::string& text);

		/**
		 * @brief Sets the keyboard input character for the shortcut
		 * @param input The key character (e.g., "S", "C", "Enter")
		 * @return Reference to this tooltip for method chaining
		 */
		ToolTip& keyInput(const std::string& input);

		/**
		 * @brief Sets multiple keyboard modifier keys for the shortcut
		 * @tparam Args Template parameter pack for Modifier objects
		 * @param args Variable number of Modifier enums
		 * @return Reference to this tooltip for method chaining
		 */
		template<typename... Args,
			typename = typename std::enable_if<conjunction<std::is_same<Args, ToolTip::Modifier>...>::value>::type>
			ToolTip& keyModifiers(Args... args);

	protected:
		/**
		 * @brief Internal template method to process multiple modifier arguments
		 * @tparam Arg First argument type
		 * @tparam Rest Remaining argument types
		 * @param arg First argument
		 * @param rest Remaining arguments
		 */
		template<typename Arg, typename... Rest,
			typename = typename std::enable_if<conjunction<std::is_same<Arg, ToolTip::Modifier>, std::is_same<Rest, ToolTip::Modifier>...>::value>::type>
			void DoKeyModifiers(Arg arg, Rest... rest);

		/// @brief Base case for DoKeyModifiers template recursion
		void DoKeyModifiers() { }

		/// @brief Internal method to process a single modifier
		void DoKeyModifier(ToolTip::Modifier modifier);
	};

	template<typename... Args, typename>
	ToolTip& ToolTip::keyModifiers(Args... args)
	{
		DoKeyModifiers(args...);
		return *this;
	}

	template<typename Arg, typename... Rest, typename>
	void ToolTip::DoKeyModifiers(Arg arg, Rest... rest)
	{
		DoKeyModifier(arg);
		DoKeyModifiers(rest...);
	}

	/**
	 * @class enable_tooltip
	 * @brief A mixin template class that adds tooltip functionality to components.
	 * 
	 * This template class can be inherited by view components to provide
	 * tooltip attachment capabilities, allowing any component to display
	 * tooltips when hovered over.
	 * 
	 * @tparam T The type of the component that will have tooltip functionality
	 * 
	 * @example
	 * @code
	 * // Usage in a component class
	 * class MyButton : public View, public enable_tooltip<MyButton>
	 * {
	 *   // Component implementation...
	 * };
	 * 
	 * // Using the tooltip functionality
	 * MyButton()
	 *   .tooltip(ToolTip("Click to perform action"));
	 * @endcode
	 */
	template<typename T>
	class ZUI_API enable_tooltip
	{
	public:
		/**
		 * @brief Attaches a tooltip to the component
		 * @param tooltip The tooltip to attach to this component
		 * @return Reference to the component for method chaining
		 */
		T& tooltip(const ToolTip& tooltip);
	};

namespace p
{
	// Tooltip-specific pipe modifiers
	class ZUI_API tooltipText : public PipeModifierPresentation
	{
	public:
		explicit tooltipText(const std::string& text);
		~tooltipText();
	protected:
		void apply(Presentation& p) const override;
	private:
		class Impl;
		Impl* _pImpl;
	};

	class ZUI_API tooltipKeyInput : public PipeModifierPresentation
	{
	public:
		explicit tooltipKeyInput(const std::string& input);
		~tooltipKeyInput();
	protected:
		void apply(Presentation& p) const override;
	private:
		class Impl;
		Impl* _pImpl;
	};

  namespace discover {
    namespace by_view {
      namespace tooltip {
        using zui::p::tooltipText;
        using zui::p::tooltipKeyInput;
      }
    }
  }

}
}

#endif // TOOLTIP_H
