/**
 * @file ZoomSDKMeetingSmartSummaryController.h
 * @brief Interfaces for controlling and managing the Smart Summary feature in Zoom meetings.
 *
 * This header file declares the @c ZoomSDKMeetingSmartSummaryController class and related protocols and handler classes.
 * It provides APIs to check, enable, start, stop, and request the Smart Summary feature during a meeting.
 *
 * The Smart Summary feature allows meeting hosts or participants to generate an AI-based summary of the meeting content.
 */

#import <Foundation/Foundation.h>
#import <ZoomSDK/ZoomSDKErrors.h>

NS_ASSUME_NONNULL_BEGIN
/**
 * @class ZoomSDKMeetingSmartSummaryPrivilegeHandler
 * @brief Handler class to process smart summary start requests.
 */
@interface ZoomSDKMeetingSmartSummaryPrivilegeHandler : NSObject
/**
 * @brief Agree to the start smart summary request.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)accept;
/**
 * @brief Decline the start smart summary request.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)decline;
/**
 * @brief Ignore the start smart summary request.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)ignore;
@end

/**
 * @protocol ZoomSDKMeetingSmartSummaryControllerDelegate
 * @brief Protocol to receive smart summary events during the meeting.
 */
@protocol ZoomSDKMeetingSmartSummaryControllerDelegate <NSObject>
/**
 * @brief Callback event when smart summary status changes.
 * @param isStart YES if the smart summary is started, NO otherwise.
 */
- (void)onSmartSummaryStatusChanged:(BOOL)isStart;
/**
 * @brief Callback event when a user requests the host to start smart summary.
 * @param senderID The user who requests the host to start smart summary.
 * @param handler The handler to handle the smart summary start request.
 */
- (void)onSmartSummaryPrivilegeRequested:(unsigned int)senderID requestHandler:(ZoomSDKMeetingSmartSummaryPrivilegeHandler*)handler;
/**
 * @brief Callback event when the host handle the smart summary request.
 * @param timeout YES if the host doesn't handle the request until timeout, NO otherwise.
 * @param decline YES if the host declines the request, NO if the host agrees to the request.
 */
- (void)onSmartSummaryStartReqResponse:(BOOL)timeout isDecline:(BOOL)decline;
/**
 * @brief Callback event when the user needs to go to the web to enable the smart summary feature. The interface onlies be triggered for Businessbiz accounts. And the smart summary feature is enabled for the future meeting, but not enable the current meeting. If the user enables the setting in web.
 */
- (void)onEnableSmartSummary;

@end


/**
 * @class ZoomSDKMeetingSmartSummaryController
 * @brief This class provides interfaces to manage and control the smart summary feature in meetings.
 */
@interface ZoomSDKMeetingSmartSummaryController : NSObject
/**
 * @brief Delegate to receive smart summary related events.
 */
@property(nonatomic,assign,nullable) id<ZoomSDKMeetingSmartSummaryControllerDelegate> delegate;
/**
 * @brief Determines if current meeting support smart summary feature.
 * @return YES if the current meeting supports the smart summary feature. Otherwise, NO.
 */
- (BOOL)isSmartSummarySupported;
/**
 * @brief Determines if the smart summary feature is enabled in the meeting.
 * @return YES if the smart summary feature is enabled. Otherwise, NO.
 */
- (BOOL)isSmartSummaryEnabled;
/**
 * @brief Whether the current user can enable the smart summary feature for the account.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)canEnableSmartSummaryFeature;
/**
 * @brief Enables the smart summary feature for the account.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableSmartSummaryFeature;
/**
 * @brief Whether the current user is able to start smart summary.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)canStartSmartSummary;
/**
 * @brief Starts smart summary.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)startSmartSummary;
/**
 * @brief Stops smart summary.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)stopSmartSummary;
/**
 * @brief Query whether smart summary is started.
 * @return YES if smart summary is started. Otherwise, NO.
 */
- (BOOL)isSmartSummaryStarted;
/**
 * @brief Whether the current user can request the admin to enable the smart summary feature for the account.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)canRequestEnableSmartSummaryFeature;
/**
 * @brief Request the admin to enable the smart summary feature for the account.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)requestEnableSmartSummaryFeature;
/**
 * @brief Whether the current user can request the host to start the smart summary for the current meeting.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)canRequestStartSmartSummary;
/**
 * @brief Request the host to start the smart summary for the current meeting.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)requestStartSmartSummary;
@end

NS_ASSUME_NONNULL_END
