/**
 * @file ZoomSDKSettingService.h
 * @brief Interface declarations for Zoom SDK setting services.
 */

#import <Foundation/Foundation.h>
#import <ZoomSDK/ZoomSDKErrors.h>
#import <ZoomSDK/ZoomSDKSettingVideoEffectController.h>
NS_ASSUME_NONNULL_BEGIN

/**
 * @brief Enumeration of the virtual background URL types.
 */
typedef enum{
    /** The learn more url type*/
    SettingComponent_VirtualBackground_Leran_More,
    /** The help url type*/
    SettingComponent_VirtualBackground_Help,
}SDKURLType;


/**
 * @class ZoomSDKAutoFramingParameter
 * @brief Parameters for auto-framing feature in Zoom SDK.
 */
@interface ZoomSDKAutoFramingParameter : NSObject
/**
 * @brief The zoom in ratio of auto-framing, valid range of values: 1~10(when mode is ZoomSDKAutoFramingMode_Center_Coordinates), 0.1~10(when mode is ZoomSDKAutoFramingMode_Face_Recognition).
 */
@property (assign,nonatomic) float ratio;
/**
 * @brief The strategy when face recognition failed.
 */
@property (assign,nonatomic) ZoomSDKFaceRecognitionFailStrategy failStrategy;
@end

/**
 * @protocol ZoomSDKSettingTestAudioDelegate
 * @brief Delegate to receive audio testing related events.
 */
@protocol ZoomSDKSettingTestAudioDelegate <NSObject>
@optional
/**
 * @brief Notification of microphone level changes when testing.
 * @param level The level of microphone.
 */
- (void)onMicLevelChanged:(unsigned int)level;
/**
 * @brief Notification of speaker level changes when testing.
 * @param level The speaker level.
 */
- (void)onSpeakerLevelChanged:(unsigned int)level;
/**
 * @brief Notify the microphone test status has changed.
 * @param status Test status of microphone.
 */
- (void)onMicTestStatusChanged:(ZoomSDKTestMicStatus)status;
/**
 * @brief Notification of speaker status changes when testing.
 * @param isTesting YES if the speaker test is in progress, NO otherwise.
 */
- (void)onSpeakerTestStatusChanged:(BOOL)isTesting;
@end


/**
 * @protocol ZoomSDKSettingAudioDeviceDelegate
 * @brief Delegate to receive audio device related events.
 */
@protocol ZoomSDKSettingAudioDeviceDelegate <NSObject>
@optional
/**
 * @brief Notify the microphone device status has changed in the meeting.
 * @param status The microphone device status.
 */
- (void)onMicDeviceStatusChanged:(ZoomSDKDeviceStatus)status;
/**
 * @brief Notify the speaker device status has changed in the meeting.
 * @param status The speaker device status.
 */
- (void)onSpeakerDeviceStatusChanged:(ZoomSDKDeviceStatus)status;
/**
 * @brief Notification that the selected microphone device is changed.
 */
- (void)onSelectedMicDeviceChanged;
/**
 * @brief Notification that the selected speaker device is changed.
 */
- (void)onSelectedSpeakerDeviceChanged;
@end


/**
 * @protocol ZoomSDKSettingVideoDelegate
 * @brief Delegate to receive video device related events.
 */
@protocol ZoomSDKSettingVideoDelegate <NSObject>
@optional
/**
 * @brief Notification of camera status changes in the meeting.
 * @param status The camera device status.
 */
- (void)onCameraStatusChanged:(ZoomSDKDeviceStatus)status;
/**
 * @brief Notification that the selected camera device is changed.
 * @param deviceID The ID of camera.
 */
- (void)onSelectedCameraChanged:(NSString*)deviceID;
@end


/**
 * @class ZoomSDKSettingTestSpeakerDeviceHelper
 * @brief Helper class to test speaker devices.
 */
@interface ZoomSDKSettingTestSpeakerDeviceHelper: NSObject
{
    id<ZoomSDKSettingTestAudioDelegate>     _delegate;
    BOOL                                    _isSpeakerInTesting;
    NSString*                               _speakerID;
}
/**
 * @brief Indicates whether speaker testing is in progress.
 */
@property (nonatomic, readwrite, assign)BOOL                    isSpeakerInTesting;

/**
 * @brief Delegate to receive audio testing events.
 */
@property(nonatomic, assign, nullable)id<ZoomSDKSettingTestAudioDelegate> delegate;

/**
 * @brief Starts playing when testing speaker. 
 * @param deviceID The speaker's device ID.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)SpeakerStartPlaying:(NSString*)deviceID;
/**
 * @brief Stops playing when testing speaker.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)SpeakerStopPlaying;
@end


/**
 * @class ZoomSDKSettingTestMicrophoneDeviceHelper
 * @brief Helper class for testing microphone devices.
 */
@interface ZoomSDKSettingTestMicrophoneDeviceHelper: NSObject
{
    id<ZoomSDKSettingTestAudioDelegate>         _delegate;
    ZoomSDKTestMicStatus         _testMicStatus;
    NSString*                    _microphoneID;
}
/**
 * @brief Delegate to receive audio testing events.
 */
@property(nonatomic, assign, nullable)id<ZoomSDKSettingTestAudioDelegate> delegate;
/**
 * @brief Starts recording when testing microphone.
 * @param deviceID The microphone's device ID.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)startRecordingMic:(NSString*)deviceID;
/**
 * @brief Stops recording when testing microphone.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)stopRecrodingMic;
/**
 * @brief Plays recorded sounds when testing microphone.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)playRecordedMic;
/**
 * @brief Stops playing recorded sounds when testing microphone.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)stopPlayRecordedMic;
/**
 * @brief Gets the status when testing microphone.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKTestMicStatus)getTestMicStatus;
@end


/**
 * @class ZoomSDKSettingTestVideoDeviceHelper
 * @brief Helper class for testing video devices.
 */
@interface ZoomSDKSettingTestVideoDeviceHelper: NSObject
{
    id<ZoomSDKSettingVideoDelegate>       _delegate;
}
/**
 * @brief Delegate to receive video device events.
 */
@property(nonatomic, assign, nullable)id<ZoomSDKSettingVideoDelegate> delegate;
/**
 * @brief Sets the frame of video preview and tells Zoom client the parent view where the video preview is drawn. 
 * @param parentView The parent view where the video preview will be drawn.
 * @param containerRect The frame displaying video preview.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)SetVideoParentView:(NSView*)parentView VideoContainerRect:(NSRect)containerRect;
/**
 * @brief Preview user's video.
 * @param deviceID The ID of camera device.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @deprecated This method is no longer used.
 */
- (ZoomSDKError)StartPreview:(NSString*)deviceID DEPRECATED_MSG_ATTRIBUTE("No longer used");

/**
 * @brief Preview user's video with default camera.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)StartPreview;
/**
 * @brief Stops previewing user's video.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)StopPreview;
@end


/**
 * @class ZoomSDKAudioStatisticsInfo
 * @brief Class representing audio transmission statistics.
 * @deprecated This class is deprecated and will be removed in a future version.
 */
DEPRECATED_MSG_ATTRIBUTE("This class is deprecated - use ZoomSDKAudioStatisticInfo instead")
@interface ZoomSDKAudioStatisticsInfo : NSObject
{
    int _frequencySend;
    int _frequencyReceive;
    int _latencySend;
    int _latencyReceive;
    int _jitterSend;
    int _jitterReceive;
    float _packageLossAvgSend;
    float _packageLossAvgReceive;
    float _packageLossMaxSend;
    float _packageLossMaxReceive;
}
/**
 * @brief Gets frequency when transferring audio data.
 * @param isSend YES if sending data, NO receiving data.
 */
- (int)getFrequency:(BOOL)isSend;
/**
 * @brief Gets latency time when transferring audio data.
 * @param isSend YES if sending data, NO receiving data.
 */
- (int)getLatency:(BOOL)isSend;
/**
 * @brief Gets jitter when transferring audio data.
 * @param isSend YES if sending data, NO receiving data.
 */
- (int)getJitter:(BOOL)isSend;
/**
 * @brief Gets the rate of losing package when transferring audio data.
 * @param isSend YES if sending data, NO receiving data.
 * @param isMax YES if the max rate of losing package, NO the average rate of losing package.
 */
- (float)getPackageLoss:(BOOL)isSend Max:(BOOL)isMax;
@end


/**
 * @class ZoomSDKVideoASStatisticsInfo
 * @brief Class representing video and sharing transmission statistics.
 * @deprecated This class is deprecated and will be removed in a future version.
 */
DEPRECATED_MSG_ATTRIBUTE("This class is deprecated - use ZoomSDKASVStatisticInfo instead")
@interface ZoomSDKVideoASStatisticsInfo : NSObject
{
    int _resolutionSend;
    int _resolutionReceive;
    int _fpsSend;
    int _fpsReceive;
    int _latencySend;
    int _latencyReceive;
    int _jitterSend;
    int _jitterReceive;
    float _packageLossAvgSend;
    float _packageLossAvgReceive;
    float _packageLossMaxSend;
    float _packageLossMaxReceive;
}
/**
 * @brief Gets latency time when transferring video or sharing data.
 * @param isSend YES if sending data, NO receiving data.
 */
- (int)getLatency:(BOOL)isSend;
/**
 * @brief Gets jitter when transferring video or sharing data.
 * @param isSend YES if sending data, NO receiving data.
 */
- (int)getJitter:(BOOL)isSend;
/**
 * @brief Gets the rate of losing package when transferring video or sharing data.
 * @param isSend YES if sending data, NO receiving data.
 * @param isMax YES if the max rate of losing package, NO the average rate of losing package.
 */
- (float)getPackageLoss:(BOOL)isSend Max:(BOOL)isMax;
/**
 * @brief Gets resolution when transferring video or sharing data.
 * @param isSend YES if sending data, NO receiving data.
 * @note height can get through (Resolution >> 16), width can get through ((Resolution << 16) >> 16).
 */
- (int)getResolution:(BOOL)isSend;
/**
 * @brief Gets the fram rate when transferring video or sharing data.
 * @param isSend YES if sending data, NO receiving data.
 */
- (int)getFPS:(BOOL)isSend;
@end

/**
 * @class ZoomSDKAudioStatisticInfo
 * @brief The meeting audio statistic information.
 */
@interface ZoomSDKAudioStatisticInfo : NSObject
/**
 * @brief The send audio frequency in kilohertz (KHz) in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  sendFrequency;
/**
 * @brief The send audio rtt in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  sendRTT;
/**
 * @brief The send audio jitter in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  sendJitter;
/**
 * @brief The average of send audio packet loss in this meeting.
 */
@property(nonatomic, assign, readonly) CGFloat    sendPacketLossAvg;
/**
 * @brief The maximum send audio packet loss in this meeting.
 */
@property(nonatomic, assign, readonly) CGFloat    sendPacketLossMax;
/**
 * @brief The send band width of audio in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  sendBandwidth;
/**
 * @brief The receive audio frequency in kilohertz (KHz) in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  recvFrequency;
/**
 * @brief  The receive audio rtt in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  recvRTT;
/**
 * @brief TThe receive audio jitter in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  recvJitter;
/**
 * @brief The average of receive audio packet loss in this meeting.
 */
@property(nonatomic, assign, readonly) CGFloat    recvPacketLossAvg;
/**
 * @brief The maximum receive audio packet loss in this meeting.
 */
@property(nonatomic, assign, readonly) CGFloat    recvPacketLossMax;
/**
 * @brief The receive band width of audio in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger    recvBandwidth;
@end

/**
 * @class ZoomSDKASVStatisticInfo
 * @brief The meeting video or share statistic information.
 */
@interface ZoomSDKASVStatisticInfo : NSObject
/**
 * @brief The send band width of video or share in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  sendBandwidth;
/**
 * @brief The send frame rate of video or share in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  sendFps;
/**
 * @brief The send video or share rtt data in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  sendRTT;
/**
 * @brief The send video or share jitter data in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  sendJitter;
/**
 * @brief The average video or share packet loss of send data in this meeting.
 */
@property(nonatomic, assign, readonly) CGFloat    sendPacketLossAvg;
/**
 * @brief The maximum video or share packet loss of send data in this meeting.
 */
@property(nonatomic, assign, readonly) CGFloat    sendPacketLossMax;
/**
 * @brief The send resolution when transferring video or sharing data.
 */
@property(nonatomic, assign, readonly) NSInteger    sendResolution;
/**
 * @brief The receive band width of video or share in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  recvBandwidth;
/**
 * @brief The receive frame rate of video or share in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  recvFps;
/**
 * @brief The receive video or share rtt data in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  recvRTT;
/**
 * @brief The receive video or share jitter data in this meeting.
 */
@property(nonatomic, assign, readonly) NSInteger  recvJitter;
/**
 * @brief The average video or share packet loss of receive data in this meeting.
 */
@property(nonatomic, assign, readonly) CGFloat    recvPacketLossAvg;
/**
 * @brief The maximum video or share packet loss of receive data in this meeting.
 */
@property(nonatomic, assign, readonly) CGFloat    recvPacketLossMax;
/**
 * @brief The receive resolution when transferring video or sharing data.
 */
@property(nonatomic, assign, readonly) NSInteger    recvResolution;
@end

/**
 * @class SDKDeviceInfo
 * @brief Class representing information about a device (microphone, speaker, camera).
 */
@interface SDKDeviceInfo : NSObject
/**
 * @brief Gets the ID of device, such as microphone, speaker, camera.
 */
- (NSString*)getDeviceID;
/**
 * @brief Gets the device name.
 */
- (NSString*)getDeviceName;
/**
 * @brief Query if the device is selected.
 */
- (BOOL)isSelectedDevice;
@end



/**
 * @class ZoomSDKAudioSetting
 * @brief Provides APIs to manage audio devices, audio volumes, and audio-related meeting features.
 */
@interface ZoomSDKAudioSetting: NSObject
{
    ZoomSDKSettingTestSpeakerDeviceHelper* _speakerTestHelper;
    ZoomSDKSettingTestMicrophoneDeviceHelper* _micTestHelper;
    id<ZoomSDKSettingAudioDeviceDelegate>       _delegate;
}
/**
 * @brief Delegate to receive audio device status changes.
 */
@property(nonatomic, assign, nullable)id<ZoomSDKSettingAudioDeviceDelegate> delegate;
/**
 * @brief Gets the object of ZoomSDKSettingTestSpeakerDeviceHelper.
 * @return If the function succeeds, it returns a ZoomSDKSettingTestSpeakerDeviceHelper object. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKSettingTestSpeakerDeviceHelper*_Nullable)getSettingSpeakerTestHelper;
/**
 * @brief Gets the object of ZoomSDKSettingTestMicrophoneDeviceHelper.
 * @return If the function succeeds, it returns a ZoomSDKSettingTestMicrophoneDeviceHelper object. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKSettingTestMicrophoneDeviceHelper*_Nullable)getSettingMicrophoneTestHelper;
/**
 * @brief Gets the list of audio device. 
 * @param mic YES if microphone device, NO if speaker device.
 * @return If the function succeeds, it returns an array containing \link SDKDeviceInfo \endlink elements. Otherwise, this function fails and returns nil.
 */
- (NSArray*_Nullable)getAudioDeviceList:(BOOL)mic;

/**
 * @brief Gets the volume of audio device. 
 * @param mic YES if the volume of microphone device, NO if speaker device.
 * @return If the function succeeds, it returns an int value(0-100). Otherwise, this function fails and returns 0.
 */
- (float)getAudioDeviceVolume:(BOOL)mic;

/**
 * @brief Sets the volume of audio device. 
 * @param mic YES if the volume of microphone device, NO if speaker device.
 * @param volume The volume of device, varies from 0 to 100.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setAudioDeviceVolume:(BOOL)mic Volume:(float)volume;

/**
 * @brief Select an audio device.
 * @param mic YES if microphone device, NO if speaker device.
 * @param deviceID The device's ID.
 * @param deviceName The device's name.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)selectAudioDevice:(BOOL)mic DeviceID:(NSString *)deviceID DeviceName:(NSString*)deviceName;

/**
 * @brief Sets to enable stereo in the meeting. 
 * @param enable YES to enable stereo, NO to disable.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableStero:(BOOL)enable;

/**
 * @brief Enables to join meeting with the audio of computer.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableAutoJoinVoip:(BOOL)enable;

/**
 * @brief Mute user's microphone when he joins the meeting with the audio of computer.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableMuteMicJoinVoip:(BOOL)enable;
/**
 * @brief Enables the feature that attendee can speak by pressing the Spacebar when he is muted.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enablePushToTalk:(BOOL)enable;

/**
 * @brief Sets whether to disable the prompt dialog of joining meeting with third party audio. 
 * @param disable YES if disabled, NO enabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)disablePromptJoinAudioDialogWhenUse3rdPartyAudio:(BOOL)disable;

/**
 * @brief Determines if the meeting supports to pop up the dialog when user joins meeting with third party audio. 
 * @return YES if supported. Otherwise, NO.
 */
- (BOOL)isSupportPromptJoinAudioDialogWhenUse3rdPartyAudio;

/**
 * @brief Determines if the dialog pops up when user joins meeting with third party audio. 
 * @return YES if the dialog will not pop up. Otherwise, NO. 
 */
- (BOOL)isPromptJoinAudioDialogWhenUse3rdPartyAudioDisable;

/**
 * @brief Enables auto-adjust microphone.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableAutoAdjustMic:(BOOL)enable;

/**
 * @brief Determines if auto-adjust microphone is enabled or not. 
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isAutoAdjustMicOn;
/**
 * @brief Determines if user joins meeting with audio on. 
 * @return YES to enable the audio. Otherwise, NO.
 */
- (BOOL)isJoinAudioWhenJoinMeetingOn;
/**
 * @brief Determines if user joins meeting with microphone muted. 
 * @return YES if muted. Otherwise, NO. 
 */
- (BOOL)isMuteMicWhenJoinMeetingOn;

/**
 * @brief Determines if stereophonic sound is enabled. 
 * @return YES if enabled. Otherwise, NO. 
 */
- (BOOL)isEnableStereoOn;
/**
 * @brief Sets whether to enable the feature that attendee can speak by pressing the Spacebar when he is muted.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isTemporarilyUnmuteOn;


/**
 * @brief Enables show original sound option in meeting UI.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableShowOriginalSoundOptionInMeetingUI:(BOOL)enable;

/**
 * @brief Determines if show original sound option in meeting UI is enabled or not.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isShowOriginalSoundOptionInMeetingUIOn;

/**
 * @brief Determines if the meeting supports show original sound option in meeting UI.
 * @return YES if supported. Otherwise, NO.
 */
- (BOOL)isSupportShowOriginalSoundOptionInMeetingUI;

/**
 * @brief Sets echo cancellation level.
 * @param level The level to be set.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setEchoCancellationLevel:(ZoomSDKAudioEchoCancellationLevel)level;
/**
 * @brief Gets echo cancellation level.
 * @return The level of echo cancellation.
 */
- (ZoomSDKAudioEchoCancellationLevel)getEchoCancellationLevel;
/**
 * @brief Enables echo cancellation.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableEchoCancellation:(BOOL)enable;

/**
 * @brief Determines if echo cancellation is enabled or not.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isEchoCancellationOn;

/**
 * @brief Determines if the meeting supports echo cancellation.
 * @return YES if supported. Otherwise, NO.
 */
- (BOOL)isSupportEchoCancellation;

/**
 * @brief Gets the level of suppressed background noise.
 * @return The level of suppressed background noise.
 */
- (ZoomSDKSuppressBackgroundNoiseLevel)getSuppressBackgroundNoiseLevel;

/**
 * @brief Sets the level of to suppress background noise.
 * @param level The level to be set.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setSuppressBackgroundNoise:(ZoomSDKSuppressBackgroundNoiseLevel)level;

/**
 * @brief Determines if used separate audio device to play ringtone simultaneously.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isAlwaysUseSeparateRingSpkOn;

/**
 * @brief Enables use separate audio device to play ringtone simultaneously.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableAlwaysUseSeparateRingSpk:(BOOL)enable;

/**
 * @brief Gets use separate audio device to play ringtone simultaneously devices.
 * @return If the function succeeds, it returns an array containing SDKDeviceInfo elements. Otherwise, this function fails and returns nil.
 */
- (NSArray *_Nullable)getRingSpkDeviceList;

/**
 * @brief Gets the volume of audio device.
 * @return If the function succeeds, it returns an float value(0.0-255.0). Otherwise returns 0.0.
 */
- (float)getRingSpkVolume;

/**
 * @brief Sets the volume of audio device.
 * @param value The value of device, varies from 0.0 to 255.0.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setRingSpkVolume:(float)value;

/**
 * @brief Select an audio device.
 * @param deviceId The ID of device.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setRingSpkDevice:(NSString*)deviceId;

/**
 * @brief Sets enable sync buttons on headset.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setSyncHeadsetButtonStatus:(BOOL)enable;

/**
 * @brief Determines if sync buttons on headset option on or off.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isSyncHeadsetButtonStatus;

/**
 * @brief Sets the "Enable Original Sound" option is high fidelity music model.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setHighFideMusicMode:(BOOL)enable;

/**
 * @brief Determines if use high fidelity music model.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isHighFideMusicMode;

/**
 * @brief Select the same audio device as system.
 * @param mic YES if microphone device, NO if speaker device.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)selectSameAudioDeviceAsSystem:(BOOL)mic;
@end

/**
 * @class ZoomSDKVideoLipSyncAvatarPreviewHelper
 * @brief Helper class to control preview of lip-sync avatars.
 */
@interface ZoomSDKVideoLipSyncAvatarPreviewHelper : NSObject

/**
 * @brief Starts to preview lip-sync avatar.
 * @param lipsyncPreview The view of show lip-sync avatar preview.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)startLipSyncAvatarPreview:(NSView *)lipsyncPreview;

/**
 * @brief Stops to preview lip-sync avatar.
 * @param lipsyncPreview The view of show lip-sync avatar preview.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)stopLipSyncAvatarPreview:(NSView *)lipsyncPreview;
@end


/**
 * @class ZoomSDKVideoPreferenceSetting
 * @brief Settings for video preference modes including frame rates and resolution adaptation.
 *
 * When custom modes are set, the developer provides the maximum and minimum frame rates. If the current bandwidth cannot maintain the minimum frame rate, the video system will drop to the next lower resolution. The default maximum and minimum frame rates for other modes are 0.
 */
@interface ZoomSDKVideoPreferenceSetting : NSObject
/**
 * @brief Preferred video mode.
 */
@property (nonatomic, assign, readwrite) ZoomSDKVideoPreferenceMode mode;
    
/**
 * @brief 0 for the default value, minimumFrameRate should be less than maximumFrameRate,
 * @note range: from 0 to 30. Out of range for frame rate will use default frame rate of Zoom.
 */
@property (nonatomic, assign, readwrite) unsigned int minimumFrameRate;
    
/**
 * @brief 0 for the default value, maximumFrameRate should be less and equal than 30,
 * @note range: from 0 to 30. Out of range for frame rate will use default frame rate of Zoom.
 */
@property (nonatomic, assign, readwrite) unsigned int maximumFrameRate;
@end


/**
 * @class ZoomSDKVideoSetting
 * @brief Interface for video settings including camera selection, video effects, beauty mode, and auto-framing.
 */
@interface ZoomSDKVideoSetting: NSObject
{
    ZoomSDKSettingTestVideoDeviceHelper* settingVideoTestHelper;
}
/**
 * @brief Gets the object to test video device.  
 * @return If the function succeeds, it returns a ZoomSDKSettingTestVideoDeviceHelper object. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKSettingTestVideoDeviceHelper*_Nullable)getSettingVideoTestHelper;

/**
 * @brief Gets the list of camera device.
 * @return If the function succeeds, it returns an array containing SDKDeviceInfo element. Otherwise, this function fails and returns nil.
 */
- (NSArray*_Nullable)getCameraList;

/**
 * @brief Select a camera.
 * @param deviceID The ID of camera.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error. 
 */
- (ZoomSDKError)selectCamera:(NSString*)deviceID;

/**
 * @brief Determines if mirror effect is enabled. 
 * @return YES if enabled. Otherwise, NO. 
 */
- (BOOL)isMirrorEffectEnabled;

/**
 * @brief Sets to enable or disable mirror effect. 
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error. 
 */
- (ZoomSDKError)enableMirrorEffect:(BOOL)enable;

/**
 * @brief Determines if facial beauty effect is enabled. 
 * @return YES if enabled. Otherwise, NO. 
 */
- (BOOL)isBeautyFaceEnabled;

/**
 * @brief Enable or disable facial beauty effect.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error. 
 */
- (ZoomSDKError)enableBeautyFace:(BOOL)enable;

/**
 * @brief Gets beauty face value.
 * @return If the function succeeds, it returns an int value(0-100). Otherwise returns 0.
 */
- (int)getBeautyFaceValue;

/**
 * @brief Sets beauty face value.
 * @param value The value type is int, varies from 0 to 100.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setBeautyFaceValue:(int)value;
/**
 * @brief Turn off the participant's video when he joins meeting. 
 * @param disable YES if the video is turned off, NO otherwise. 
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error. 
 */
- (ZoomSDKError)disableVideoJoinMeeting:(BOOL)disable;

/**
 * @brief Display or Hide username on the video window. 
 * @param display YES to showing always username on the video window, NO otherwise.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error. 
 */
- (ZoomSDKError)displayUserNameOnVideo:(BOOL)display;

/**
 * @brief Enables or disable HD video. 
 * @param enable YES if enabled, NO if disabled. 
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error. 
 */
- (ZoomSDKError)enableCatchHDVideo:(BOOL)enable;
/**
 * @brief Sets to capture video ratio: original or 16:9. 
 * @param originalSize YES if original video, NO 16:9.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error. 
 */
- (ZoomSDKError)onVideoCaptureOriginalSizeOr16To9:(BOOL)originalSize;
/**
 * @brief Enables to spotlight the current user's video in the main interface when he speaks in the meeting. 
 * @param enable YES if spotlighting always the current user's video, NO otherwise.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)onSpotlightMyVideoWhenISpeaker:(BOOL)enable;
/**
 * @brief Enables or disable to show the participants in Gallery View up to 49 per screen.
 * @param enable YES indicates to show the participants in Gallery View up to 49 per screen, NO otherwise.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error. 
 */
- (ZoomSDKError)onDisplayUpTo49InWallView:(BOOL)enable;
/**
 * @brief Enables or disable to hide the non-video participants.
 * @param hide YES if hiding, NO if displaying.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error. 
 */
- (ZoomSDKError)hideNoVideoUser:(BOOL)hide;
/**
 * @brief Determines if non-video participant is hided or not. 
 * @return YES if enabled. Otherwise, NO. 
 */
- (BOOL)isHideNoVideoUser;
/**
 * @brief Query if original size of video is enabled.  
 * @return YES if enabled. Otherwise, NO. 
 */
- (BOOL)isCaptureOriginalSize;
/**
 * @brief Determines if spotlighting the current user's video is enabled. 
 * @return YES if enabled. Otherwise, NO. 
 */
- (BOOL)isSpotlightMyVideoOn;
/**
 * @brief Determines if the current user's video is muted when he joins meeting. 
 * @return YES if muted. Otherwise, NO. 
 */
- (BOOL)isMuteMyVideoWhenJoinMeetingOn;
/**
 * @brief Determines if displaying screen name of the user is enabled.
 * @return YES if enabled. Otherwise, NO. 
 */
- (BOOL)isdisplayUserNameOnVideoOn;
/**
 * @brief Determines if it is able to display up to 49 participants in video wall mode.
 * @return YES if able. Otherwise, NO.  
 */
- (BOOL)isCanDisplayUpTo49InWallView;
/**
 * @brief Determines whether to display up to 49 participants in video wall mode. 
 * @return YES if enabled. Otherwise, NO. 
 */
- (BOOL)isDisplayUpTo49InWallViewOn;
/**
 * @brief Determines if HD video is enabled. 
 * @return YES if enabled. Otherwise, NO. 
 */
- (BOOL)isCatchHDVideoOn;
/**
 * @brief Determines if adjustion for low light.
 * @return The ZoomSDKSettingVideoLightAdaptionModel.
 */
- (ZoomSDKSettingVideoLightAdaptionModel)getLightAdjustModel;
/**
 * @brief Sets the way to adjust the low light.
 * @param model The model to be set.
 * @param value The value type is int, varies from 0 to 100.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setLightAdaptionModel:(ZoomSDKSettingVideoLightAdaptionModel)model LightAdaptionManualValue:(int)value;
/**
 * @brief Gets the setted's value.
 * @return If the function succeeds, it returns an int value(0-100). Otherwise returns 0.
 */
- (int)getLightAdaptionManualValue;

/**
 * @brief Determines whether to hardware acceleration for video receive.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isHardwareAccelerationForVideoReceiveOn;

/**
 * @brief Enables or disable hardware acceleration for video receive.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableHardwareAccelerationForVideoReceive:(BOOL)enable;

/**
 * @brief Determines whether to de-noise.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isTemporalDeNoiseOn;

/**
 * @brief Enables or disable de-noise.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableTemporalDeNoise:(BOOL)enable;

/**
 * @brief Determines if optimizing received video quality is supported.
 * @return YES if it is supported, Otherwise, NO.
 */
- (BOOL)isOptimizeVideoQualitySupported;

/**
 * @brief Determines if optimizing received video quality is enabled.
 * @return YES if optimization is enabled. Otherwise, NO.
 */
- (BOOL)isOptimizeVideoQualityEnabled;

/**
 * @brief Enables or disable optimizing received video quality when facing network issues for a variety of reasons. Applies to the window in focus: speaker view, pinned / spotlighted videos, gallery view with a small number of videos
 * @param enable YES to enable this feature, NO to disable this feature.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableOptimizeVideoQuality:(BOOL)enable;

/**
 * @brief Enables or disable to stop incoming video.
 * @param enable YES indicates to enable to stop incoming video, NO to disable.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableStopIncomingVideo:(BOOL)enable;

/**
 * @brief Gets the flag to enable or disable to stop in coming video.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isStopIncomingVideoEnabled;

/**
 * @brief Enables or disable to hide the userself's view.
 * @param enable YES indicates to enable to hide the userself's view, NO to disable.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableHideSelfView:(BOOL)enable;

/**
 * @brief Gets the flag to enable or disable to hide userself's view.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isHideSelfViewEnabled;

/**
 * @brief Starts my video auto-framing.
 * @param mode The auto-framing mode.
 * @param parameter The setting of video auto-framing.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableVideoAutoFraming:(ZoomSDKAutoFramingMode)mode setting:(ZoomSDKAutoFramingParameter *)parameter;

/**
 * @brief Stops video auto-framing.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)disableVideoAutoFraming;

/**
 * @brief Determines whether auto-framing is enabled.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isVideoAutoFramingEnabled;

/**
 * @brief Gets the auto-framing mode.
 * @return The auto-framing mode.
 */
- (ZoomSDKAutoFramingMode)getVideoAutoFramingMode;

/**
 * @brief Gets the auto-framing parameter.
 * @param mode The auto-framing mode.
 * @return If the function succeeds, it returns the auto-framing parameter. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKAutoFramingParameter *_Nullable)getVideoAutoFramingSettingWithMode:(ZoomSDKAutoFramingMode)mode;

/**
 * @brief Sets the mode of auto-framing when auto-framing is enabled.
 * @param mode The mode of auto-framing.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setVideoAutoFramingMode:(ZoomSDKAutoFramingMode)mode;

/**
 * @brief Sets the zoom in ratio of auto-framing when auto-framing is enabled.
 * @param ratio The zoom in ratio, effective range: 0.7 < ratio < 2.1.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setVideoAutoFramingRatio:(float)ratio;

/**
 * @brief Sets the fail strategy of face recognition when auto-framing is enabled(mode is @c ZoomSDKAutoFramingMode_Face_Recognition).
 * @param strategy The fail strategy of face recognition.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setFaceRecognitionFailStrategy:(ZoomSDKFaceRecognitionFailStrategy)strategy;

/**
 * @brief Sets the video quality preference that automatically adjust user's video to prioritize frame rate versus resolution based on the current bandwidth available.
 * @param preferenceSetting An instance of ZoomSDKVideoPreferenceSetting.
 * @return If the function succeeds, it returns ZMVideoSDKErrors_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setVideoQualityPreference:(ZoomSDKVideoPreferenceSetting*)preferenceSetting;
@end


/**
 * @protocol ZoomSDKSettingRecordDelegate
 * @brief Delegate to notify cloud recording storage info.
 */
@protocol ZoomSDKSettingRecordDelegate <NSObject>
@optional
/**
 * @brief Notify of cloud recording storage info.
 * @param totalSize The total size of cloud recording storage.
 * @param usedSize The used size of cloud recording storage.
 * @param allowExceedStorage YES if exceeding storage is allowed, NO otherwise.
 */
- (void)onNotifyCloudRecordingStorageInfo:(long long)totalSize usedSize:(long long)usedSize isAllowExceedStorage:(BOOL)allowExceedStorage;
@end


/**
 * @class ZoomSDKRecordSetting
 * @brief Interface for recording settings.
 */
@interface ZoomSDKRecordSetting: NSObject
{
    id<ZoomSDKSettingRecordDelegate>       _delegate;
}
/**
 * @brief Delegate object for receiving recording setting events.
 */
@property(nonatomic, assign, nullable)id<ZoomSDKSettingRecordDelegate> delegate;
/**
 * @brief Sets the path for saving the meeting recording file. 
 * @param path The path for saving the meeting recording file.
 * @return If the function succeeds, it returns the @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note The parameter 'path' must already be present, or the path cannot be set successfully.
 */
- (ZoomSDKError)setRecordingPath:(NSString*)path;

/**
 * @brief Gets the current meeting's recorded path.
 * @return If the function succeeds, it returns the recording path. Otherwise, this function fails and returns nil.
 */
- (NSString*_Nullable)getRecordingPath;

/**
 * @brief Determines if choose recording path when meeting ended is enabled or not.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isEnableChooseRecordingPathWhenMeetingEnd;

/**
 * @brief Enables or disable choose recording path when meeting ended.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)chooseRecordingPathWhenMeetingEnd:(BOOL)enable;

/**
 * @brief Determines if record audio for every attendee individually is enabled or not.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isEnableRecordAudioForEveryAttendeeIndividually;

/**
 * @brief Enables or disable record audio for every attendee individually.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)recordAudioForEveryAttendeeIndividually:(BOOL)enable;

/**
 * @brief Determines if optimize for third party video editor is enabled or not.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isEnableOptimizeFor3PartyVideoEditor;

/**
 * @brief Enables or disable optimize for third party video editor.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)OptimizeFor3PartyVideoEditor:(BOOL)enable;

/**
 * @brief Determines if add timestamp for recording is enabled or not.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isEnableAddTimestampForRecording;

/**
 * @brief Enables or disable add timestamp for recordin.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)addTimestampForRecording:(BOOL)enable;

/**
 * @brief Determines if recording during screen sharing is enabled or not.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isEnableRecordDuringScreenSharing;

/**
 * @brief Enables or disable record during screen sharing .
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)recordDuringScreenSharing:(BOOL)enable;

/**
 * @brief Determines if display video next to share contents in recording file is enabled or not.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isEnableDisplayVideoNextToShareContentsInRecordingFile;

/**
 * @brief Enables or disable display video next to share contents in recording file.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)displayVideoNextToShareContentsInRecordingFile:(BOOL)enable;

/**
 * @brief Determines if the user have privilege to get cloud recording storage info.
 * @return YES if can get info. Otherwise, NO.
 */
- (BOOL)canGetCloudRecordingStorageInfo;

/**
 * @brief Call to get cloud recording storage info if the user has the privilege to get cloud recording storage info.
 * @return If the function succeeds, it returns the @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note The cloud recording storage info will be notified to user through delegate ZoomSDKSettingRecordDelegate when you have call this api successfully.
 */
- (ZoomSDKError)getCloudRecordingStorageInfo;

/**
 * @brief Determines if the user have privilege to get cloud recording management URL.
 * @return YES if can get url. Otherwise, NO.
 */
- (BOOL)canGetRecordingManagementURL;

/**
 * @brief Call to get cloud recording management URL if the user has the privilege to get cloud recording management URL.
 * @return If the function succeeds, it returns the cloud recording management URL. Otherwise, this function fails and returns nil.
 */
- (NSString*_Nullable)getRecordingManagementURL;

/**
 * @brief Sets the cloud recording manage url in the recording setting page.
 * @param recordingMgrUrl Specify the cloud recording manage url in the recording setting page.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setCustomizedCloudRecordingMgrUrl:(NSString *)recordingMgrUrl;
@end


/**
 * @class ZoomSDKWallpaperItem
 * @brief Represents a wallpaper item.
 */
@interface ZoomSDKWallpaperItem : NSObject
/**
 * @brief Wall-paper ID
 */
@property (copy, nonatomic, readonly, nullable) NSString *wallpaperId;
/**
 * @brief The title.
 */
@property (copy, nonatomic, readonly, nullable) NSString *title;
/**
 * @brief Thumbnail path.
 */
@property (copy, nonatomic, readonly, nullable) NSString *thumbnailPath;
/**
 * @brief Full image path.
 */
@property (copy, nonatomic, readonly, nullable) NSString *path;
/**
 * @brief 0 ~ 255, -1 means no transparency. Only possible in Meeting wall-paper.
 */
@property (assign, nonatomic, readwrite) int transparency;
/**
 * @brief Layout mode of the wall-paper.
 */
@property (assign, nonatomic, readwrite) ZoomSDKWallpaperLayoutMode layoutMode;
@end


/**
 * @protocol ZoomSDKWallpaperControllerDelegate
 * @brief Delegate for wallpaper events.
 */
@protocol ZoomSDKWallpaperControllerDelegate <NSObject>
/**
 * @brief The personal wall-paper change.
 * @param item The changed item.
 */
- (void)onPersonalWallpaperChanged:(ZoomSDKWallpaperItem * _Nullable)item;
/**
 * @brief The status of personal wall-paper download.
 * @param status The wall-paper image download status.
 * @param wallpaperId The downloaded wall-paper image ID.
 */
- (void)onPersonalWallpaperImageDownloadStatus:(ZoomSDKWallpaperSettingStatus)status wallpaper:(NSString *)wallpaperId;
/**
 * @brief The meeting wall-paper change.
 * @param item The  changed item.
 */
- (void)onMeetingWallpaperChanged:(ZoomSDKWallpaperItem * _Nullable)item;
/**
 * @brief The status of meeting wall-paper download.
 * @param status The wall-paper image download status.
 * @param wallpaperId The download wall-paper image ID.
 */
- (void)onMeetingWallpaperImageDownloadStatus:(ZoomSDKWallpaperSettingStatus)status wallpaper:(NSString *)wallpaperId;
@end


/**
 * @class ZoomSDKWallpaperController
 * @brief Controller for managing wallpapers.
 */
@interface ZoomSDKWallpaperController : NSObject
/**
 * @brief Delegate object for handling wallpaper events.
 */
@property(nonatomic, assign, nullable)id<ZoomSDKWallpaperControllerDelegate> delegate;

/**
 * @brief Determines if the personal wall-paper feature enabled by OP.
 * @return YES if the feature enabled. Otherwise, NO.
 */
- (BOOL)isPersonalWallpaperEnabled;
/**
 * @brief Gets the current user's persional wall-paper config.
 * @return If the function succeeds, it returns the current user's personal wall-paper. Otherwise, this function fails and returns nil.
 * @note Only login user have this config, if select None, the wall-paper ID is empty.
 */
- (ZoomSDKWallpaperItem *)getCurrentPersonalWallpaperItem;
/**
 * @brief Gets the current user's persional wall-paper config list.
 * @return If the function succeeds, it returns the current user's personal wall-paper list. Otherwise, this function fails and returns nil.
 * @note Only login user have this config list.
 */
- (NSArray <ZoomSDKWallpaperItem *>* _Nullable)getPersonalWallpaperList;
/**
 * @brief Sets the current user's persional wall-paper config.
 * @param item Wall-paper item need to set.
 * @return If the function succeeds, it returns the @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note Only login user have this config list.
 */
- (ZoomSDKError)setPersonalWallpaper:(ZoomSDKWallpaperItem *)item;
/**
 * @brief Gets the persional wall-paper config by wall-paper ID.
 * @return If the function succeeds, it returns the personal wall-paper with the wall-paper ID. Otherwise, this function fails and returns nil.
 * @warning Only login user have this feature.
 */
- (ZoomSDKWallpaperItem * _Nullable)getPersonalWallpaperItemById:(NSString *)wallpaperId;
/**
 * @brief Determines if the meeting wall-paper feature enabled by OP.
 * @return YES if the feature enabled. Otherwise, NO.
 */
- (BOOL)isMeetingWallpaperEnabled;
/**
 * @brief Determines if meeting wall-paper thumbnail ready.
 * @return YES if ready. Otherwise, NO.
 */
- (BOOL)isMeetingWallpaperThumbsReady;
/**
 * @brief Gets the meeting wall-paper config.
 * @return If the function succeeds, it returns the current used meeting wall-paper. Otherwise, this function fails and returns nil.
 * @warning if select None, the wall-paper ID is empty.
 */
- (ZoomSDKWallpaperItem * _Nullable)getCurrentMeetingWallpaperItem;
/**
 * @brief Gets the meeting wall-paper config list.
 * @return If the function succeeds, it returns the meeting wall-paper list. Otherwise, this function fails and returns nil.
 */
- (NSArray <ZoomSDKWallpaperItem *>* _Nullable)getMeetingWallpaperList;
/**
 * @brief Sets the meeting wall-paper config.
 * @param item Wall-paper item need to set.
 * @return SDK error of the set opration. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setMeetingWallpaper:(ZoomSDKWallpaperItem *)item;
/**
 * @brief Gets the meeting wall-paper config by wall-paper ID.
 * @return If the function succeeds, it returns the meeting wall-paper with the wall-paper ID. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKWallpaperItem *)getMeetingWallpaperById:(NSString *)wallpaperId;
@end


/**
 * @class ZoomSDKGeneralSetting
 * @brief General settings interface for controlling meeting behaviors and UI customizations.
 */
@interface ZoomSDKGeneralSetting: NSObject
/**
 * @brief Enables or disable meeting settings by command.
 * @param enable YES to enable, NO otherwise.
 * @param cmd An enumeration of commands that you can enable or disable them in the meeting.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableMeetingSetting:(BOOL)enable SettingCmd:(MeetingSettingCmd)cmd;
/**
 * @brief Sets custom link for invitation.
 * @param inviteURL The URL for invitation by which user can join meeting.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error. 
 */
- (ZoomSDKError)setCustomInviteURL:(NSString*)inviteURL;

/**
 * @brief Custom support URL.
 * @param feedbackURL Support URL.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @deprecated This method is no longer used.
 */
- (ZoomSDKError)setCustomFeedbackURL:(NSString*)feedbackURL DEPRECATED_MSG_ATTRIBUTE("No longer used");

/**
 * @brief Hides setting components. 
 * @param component An enumeration of components for controlling meeting.
 * @param hide YES to hide, NO to show.
 */
- (void)hideSettingComponent:(SettingComponent)component hide:(BOOL)hide;

/**
 * @brief Sets default URL for setting components.
 * @param urlType A enum specify the url you want to modify.
 * @return If the function succeeds, it returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setCustomURL:(SDKURLType)urlType urlString:(NSString*)urlString;

/**
 * @brief Gets the current status of show meeting time
 * @return Yes is means show meeting time. Otherwise, NO.
 */
- (BOOL)isShowLockMeetingTime;

/**
 * @brief Enables or disenable to set show meeting time
 * @param enable YES means show meeting time otherwise is hidden
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableToShowMeetingTime:(BOOL)enable;

/**
 * @brief Determines if can copy invite url after start meeting.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isEnableCopyInviteURL;

/**
 * @brief To set meeting invite url can copy.
 * @param enable YES if can copy, NO otherwise.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setCopyMeetingInviteURL:(BOOL)enable;
/**
 * @brief Determines if can comfire when leaving meeting.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isEnableConfirmLeavingMeeting;

/**
 * @brief To set confirm when user leaving meeting.
 * @param enable Yes means will confirm when user leaving meeting.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setConfirmLeavingMeeting:(BOOL)enable;

/**
 * @brief To set the appearance of ui.
 * @param appearance The enum of appearance.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setUIAppearance:(ZoomSDKUIAppearance)appearance;

/**
 * @brief Gets the UI appearance type.
 * @return The UI appearance type.
 */
- (ZoomSDKUIAppearance)getUIAppearance;

/**
 * @brief Determines if support set ui theme.
 * @return If support, it returns YES. Otherwise, NO.
 */
- (BOOL)isSupportSetUITheme;

/**
 * @brief Sets the theme of ui.
 * @param theme The enum of theme.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setUITheme:(ZoomSDKUITheme)theme;

/**
 * @brief Query the theme of ui.
 * @return The theme of ui.
 */
- (ZoomSDKUITheme)getUITheme;

/**
 * @brief Stops my video and audio when my display is off or screen saver begins.
 * @param enable YES is means mute video and aodio when lock screen,otherewise not, NO otherwise.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setMuteVideoAndAudioWhenLockScreen:(BOOL)enable;

/**
 * @brief Determines if mute video and aodio when lock screen.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isMutedVideoAndAudioWhenLockScreen;

/**
 * @brief Sets the reaction's skintone.
 * @param skinTone The enum of skintone.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setReactionSkinTone:(ZoomSDKEmojiReactionSkinTone)skinTone;

/**
 * @brief Gets the reaction's skintone.
 * @return The current's value reaction skintone.
 */
- (ZoomSDKEmojiReactionSkinTone)getReactionSkinTone;

/**
 * @brief Hides checkbox of automatically copy invite link when meeting start.
 * @param hide YES to hide the checkbox, NO otherwise.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)hideAutoCopyInviteLinkCheckBox:(BOOL)hide;
/**
 * @brief Mute attendees when they join the meeting.
 * @param bEnable YES if attendees join the meeting is muted, NO otherwise.
 * @param allow YES if attendee can unmute self, NO otherwise.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableMuteOnEntry:(BOOL)bEnable allowUnmuteBySelf:(BOOL)allow;
/**
 * @brief Gets the object of set wallpaper controller.
 * @return If the function succeeds, it returns a ZoomSDKWallpaperController object. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKWallpaperController*)getWallpaperController;
@end


/**
 * @class ZoomSDKStatisticsSetting
 * @brief Interface for obtaining meeting statistics.
 */
@interface ZoomSDKStatisticsSetting: NSObject
/**
 * @brief Gets the connection type of current meeting.
 * @return An enumeration of connection type.
 */
- (SettingConnectionType)getSettingConnectionType;

/**
 * @brief Gets network type of current meeting.
 * @return An enumeration of network type.
 */
- (SettingNetworkType)getSettingNetworkType;
/**
 * @brief Gets the proxy address of current meeting.
 * @return Proxy address if the meeting uses a proxy. Otherwise, this function fails and returns nil.
 */
- (NSString*_Nullable)getProxyAddress;
/**
 * @brief Gets audio statistic information of the current meeting.
 * @return If the function succeeds, it returns an object of ZoomSDKAudioStatisticsInfo. Otherwise, this function fails and returns nil.
 * @deprecated This method is no longer used.
 */
- (ZoomSDKAudioStatisticsInfo*_Nullable)getAudioStatisticsInfo DEPRECATED_MSG_ATTRIBUTE("Use -getAudioStatisticInfo instead");

/**
 * @brief Gets Video or AS statistic information of the current meeting.
 * @param isVideo YES to get video statistic information, NO to get AS statistics information.
 * @return If the function succeeds, it returns an object of ZoomSDKVideoASStatisticsInfo. Otherwise, this function fails and returns nil.
 * @deprecated This method is no longer used.
 */
- (ZoomSDKVideoASStatisticsInfo*_Nullable)getVideoASStatisticsInfo:(BOOL)isVideo DEPRECATED_MSG_ATTRIBUTE("Use -getVideoStatisticInfo and -getShareStatisticInfo instead");

/**
 * @brief Gets meeting audio statistics information.
 * @return If the function succeeds, it returns an object of ZoomSDKAudioStatisticInfo. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKAudioStatisticInfo*_Nullable)getAudioStatisticInfo;

/**
 * @brief Gets meeting video statistics information.
 * @return If the function succeeds, it returns an object of ZoomSDKASVStatisticInfo. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKASVStatisticInfo*_Nullable)getVideoStatisticInfo;

/**
 * @brief Gets meeting share statistics information.
 * @return If the function succeeds, it returns an object of ZoomSDKASVStatisticInfo. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKASVStatisticInfo*_Nullable)getShareStatisticInfo;
@end


/**
 * @class ZoomSDKVirtualBGImageInfo
 * @brief Information of a virtual background image or video item.
 */
@interface ZoomSDKVirtualBGImageInfo: NSObject

/**
 * @brief Determines if it is the selected virtual background image.
 * @return YES if is the selected virtual background image. Otherwise, NO.
 */
- (BOOL)isSelected;
/**
 * @brief Gets file path of the virtual background image.
 * @return If the function succeeds, it returns the image file path. Otherwise, this function fails and returns nil.
 */
- (NSString*_Nullable)getImageFilePath;
/**
 * @brief Gets image file name of the virtual background image.
 * @return If the function succeeds, it returns the image file name. Otherwise, this function fails and returns nil.
 */
- (NSString*_Nullable)getImageName;
/**
 * @brief Determines if the selected virtual background is video.
 * @return YES if is the selected virtual background is video. Otherwise, NO.
 */
- (BOOL)isVideo;

/**
 * @brief Determines if the  virtual background item can be deleted.
 * @return YES if is the selected virtual background can be deleted. Otherwise, NO.
 */
- (BOOL)isAllowDelete;
@end


/**
 * @class ZoomSDKVideoFilterItemInfo
 * @brief Information of a video filter (face effect) item.
 */
@interface ZoomSDKVideoFilterItemInfo: NSObject

/**
 * @brief Determines if it is the selected virtual background image.
 * @return YES if is the selected virtual background image. Otherwise, NO.
 */
- (BOOL)isSelected;

/**
 * @brief Gets file path of the virtual background image.
 * @return If the function succeeds, it returns the image file path. Otherwise, this function fails and returns nil.
 */
- (NSString*_Nullable)getImageFilePath;

/**
 * @brief Gets image file name of the virtual background image.
 * @return If the function succeeds, it returns the image file name. Otherwise, this function fails and returns nil.
 */
- (NSString*_Nullable)getImageName;

/**
 * @brief Gets the virtual background image or video item's type.
 * @return The video effect type.
 */
- (ZoomSDKVideoEffectType)getType;

/**
 * @brief Gets the virtual background image or video item's index.
 * @return If the function succeeds, the return value is the index. Otherwise, returns -1.
 */
- (int)getIndex;
@end


/**
 * @protocol ZoomSDKVirtualBackgroundSettingDelegate
 * @brief Protocol for receiving virtual background setting events.
 */
@protocol ZoomSDKVirtualBackgroundSettingDelegate <NSObject>
@optional
/**
 * @brief Notify the default virtual background image have been downloaded from web.
 * @param filePath The file's path.
 */
- (void)onVBImageDidDownloaded:(NSString*)filePath;

/**
 * @brief Notify the virtual background was updated with selected color.
 * @param selectedColor The selected color.
 */
- (void)onGreenVBDidUpdateWithReplaceColor:(NSColor*)selectedColor;

/**
 * @brief Notify the selected virtual background image has been changed, user can get the new selected image through image list.
 */
- (void)onSelectedVBImageChanged;

/**
 * @brief Notify the result of adding video virtual background.
 * @param success YES if successfully added, NO otherwise.
 * @param error If failed adding the video virtual background, the error will be notified.
 */
- (void)onVBVideoUploadedResult:(BOOL)success failedError:(ZoomSDKSettingVBVideoError)error;

/**
 * @brief Callback event of notification that the thumbnails of all video filter items have been downloaded.
 */
- (void)onVideoFilterItemThumnailsDownloaded;

/**
 * @brief Callback event of notification that the selected video filter item is downloading.
 * @param type The selected video filter item's type.
 * @param index The selected video filter item's index.
 */
- (void)onVideoFilterItemDataDownloading:(ZoomSDKVideoEffectType)type index:(int)index;

/**
 * @brief Callback event of notification that the selected video filter item whether has been downloaded successfully.
 * @param ready YES indicates the selected video filter item has been downloaded successfully, NO otherwise.
 * @param type The selected video filter item's type.
 * @param index The selected video filter item's index.
 */
- (void)onVideoFilterItemDataDownloaded:(BOOL)ready type:(ZoomSDKVideoEffectType)type index:(int)index;
@end


/**
 * @class ZoomSDKVirtualBackgroundSetting
 * @brief Interface for configuring virtual background and video filter settings.
 */
@interface ZoomSDKVirtualBackgroundSetting: NSObject
{
    id<ZoomSDKVirtualBackgroundSettingDelegate>       _delegate;
}
@property(nonatomic, assign, nullable)id<ZoomSDKVirtualBackgroundSettingDelegate> delegate;
/**
 * @brief Gets the object to video device test helper.
 * @return If the function succeeds, it returns a ZoomSDKSettingTestVideoDeviceHelper object. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKSettingTestVideoDeviceHelper*_Nullable)getSettingVideoTestHelper;

/**
 * @brief Determines if support virtual background feature.
 * @return YES if is support. Otherwise, NO.
 */
- (BOOL)isSupportVirtualBG;

/**
 * @brief Determines if meeting supports the video virtual video background feature.
 * @return YES if is support. Otherwise, NO.
 */
- (BOOL)isSupportVirtualBackgroundVideo;

/**
 * @brief Determines if the machine can support the smart virtual background feature.
 * @return YES if is support. Otherwise, NO.
 */
- (BOOL)isDeviceSupportSmartVirtualBG;

/**
 * @brief Determines if the using green screen option is enabled.
 * @return YES if is enabled. Otherwise, NO.
 */
- (BOOL)isUsingGreenScreenOn;

/**
 * @brief Enables or disable the using green screen option.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setUsingGreenScreen:(BOOL)bUse;

/**
 * @brief Adds virtual background image.
 * @param filePath The image's file path user want to add.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)addBGImage:(NSString*)filePath;

/**
 * @brief Adds virtual background video.
 * @param filePath The video's file path user want to add.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)addBGVideo:(NSString*)filePath;

/**
 * @brief Removes virtual background item.
 * @param virtualBGImageInfo The ZoomSDKVirtualBGImageInfo object user want to remove.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)removeBGItem:(ZoomSDKVirtualBGImageInfo *)virtualBGImageInfo;

/**
 * @brief Gets the array of virtual background video item.
 * @return If the function succeeds, it returns the NSArray of video item list. Otherwise, this function fails and returns nil.
 */
- (NSArray*_Nullable)getBGItemList;

/**
 * @brief Use the specify image as selected virtual background images.
 * @param item The virtualbackground image or video item's path user want to select.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)useBGItem:(ZoomSDKVirtualBGImageInfo*)item;

/**
 * @brief Gets the selected replace color of virtual background images.
 * @return If the function succeeds, it returns the color. Otherwise, this function fails and returns nil.
 */
- (NSColor*_Nullable)getVBReplaceColor;

/**
 * @brief Starts selected replace color of virtual background images.
 * @return If the function succeeds, it returns the @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note The selected replace color will be notified from callback event '- (void)onSelectedVBImageChanged'.
 */
- (ZoomSDKError)startSelectReplaceVBColor;

/**
 * @brief Determines if the machine can support the smart virtual background video feature.
 * @return YES if is support. Otherwise, NO.
 */
- (BOOL)isDeviceSupportSmartVirtualBackgroundVideo;

/**
 * @brief Determines if the machine can support the green virtual background video feature.
 * @return YES if is support. Otherwise, NO.
 */
- (BOOL)isDeviceSupportGreenVirtualBackgroundVideo;

/**
 * @brief Determines if the meeting supports the adding a new virtual background item feature.
 * @return YES if is allowed. Otherwise, NO.
 */
- (BOOL)isAllowAddNewVBItem;

/**
 * @brief Determines if the meeting supports the removing a new virtual background item feature.
 * @return YES if is allowed remove. Otherwise, NO.
 */
- (BOOL)isAllowRemoveVBItem;

/**
 * @brief Determines if face makeup feature is enabled.
 * @return YES if is enabled. Otherwise, NO.
 */
- (BOOL)isVideoFilterEnabled;

/**
 * @brief Determines if support face makeup feature.
 * @return YES if is support. Otherwise, NO.
 */
- (BOOL)isSupportVideoFilter;

/**
 * @brief Gets the array of face makeup images.
 * @return If the function succeeds, it returns the NSArray of image list. Otherwise, this function fails and returns nil.
 */
- (NSArray*_Nullable)getVideoFilterItemList;

/**
 * @brief Use the specify image as selected face makeup images.
 * @param imageInfo It is video filter object.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)useVideoFilterItem:(ZoomSDKVideoFilterItemInfo*)imageInfo;
@end


/**
 * @class ZoomSDKShareScreenSetting
 * @brief Interface for configuring share screen settings.
 */
@interface ZoomSDKShareScreenSetting : NSObject

/**
 * @brief Determines if it is able to silence system notifications when sharing desktop.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isDoNotDisturbInSharingOn;

/**
 * @brief Enables or disable silence system notifications when sharing desktop.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableDoNotDisturbInSharing:(BOOL)enable;

/**
 * @brief Determines if it is able to show green border when sharing.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isGreenBorderOn;

/**
 * @brief Enables or disable show green border when sharing.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableGreenBorder:(BOOL)enable;

/**
 * @brief Determines if it is able to share selected app window only.
 * @return YES if enabled. Otherwise, NO.
 */
- (BOOL)isShareSelectedWndOnlyOn;

/**
 * @brief Enables or disable share selected app window only.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableShareSelectedWndOnly:(BOOL)enable;

/**
 * @brief Determines if it is able to using tcp connection for screen sharing.
 * @return YES if disabled. Otherwise, NO.
 */
- (BOOL)isTCPConnectionOn;

/**
 * @brief Enables or disable  use tcp connection for screen sharing.
 * @param enable YES if enabled, NO if disabled.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableTCPConnection:(BOOL)enable;

/**
 * @brief Sets screen capture mode.
 * @param mode The mode to be set.
 * @return If the function succeeds, it returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setScreenCaptureMode:(ZoomSDKScreenCaptureMode)mode;

/**
 * @brief Gets screen capture mode
 * @return The current screen capture mode.
 */
- (ZoomSDKScreenCaptureMode)getScreenCaptureMode;

/**
 * @brief Enables or disable meeting settings by command.
 * @param enable YES to enable, NO otherwise.
 * @param shareCmd Is a enumeration to set different share screen.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableSetShareScreen:(BOOL)enable  SettingCmd:(shareSettingCmd)shareCmd;

/**
 * @brief Gets the setting item current status.
 * @param sharingCmd Is a enumeration.
 * @return  If return YES is enable. Otherwise, NO.
 */
- (BOOL)isEnableToSettingShare:(shareSettingCmd)sharingCmd;

/**
 * @brief Get current FPS value.
 * @return The fps value.
 */
- (int)getLimitFPSValue;

/**
 * @brief Set the value of fps limit.
 * @param value Is a number of user to set.
 */
- (ZoomSDKError)setLimitedFPSValue:(ZoomSDKFPSValue)value;
/**
 * @brief  Enables to set limited fps.
 * @return Return YES is enable. Otherwise, NO.
 */
- (BOOL)isEnableToSetLimitFPS;

/**
 * @brief Enables or disenable to set Limit fps value.
 * @param enable Yes means to set otherwise is not.
 */
- (ZoomSDKError)setEnableLimitFPS:(BOOL)enable;

/**
 * @brief Determines if support show zoom meeting window when share.
 * @return YES if support. Otherwise, NO.
 */
- (BOOL)isSupportShowZoomWindowWhenShare;
/**
 * @brief Sets enable show zoom meeting window when share.
 * @param show YES to show, NO otherwise.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setShowZoomWindowWhenShare:(BOOL)show;

/**
 * @brief Determines if show zoom meeting window when share.
 * @return YES to show. Otherwise, NO.
 */
- (BOOL)isShowZoomWindowWhenShare;
/**
 * @brief Determines if enable share desktop.
 * @return YES to enable. Otherwise, NO.
 */
- (BOOL)isShareDesktopEnabled;
/**
 * @brief Sets the share option when share appliaction.
 * @param shareOption It is a enumeration of share option.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setShareOptionWhenShareApplication:(ZoomSDKSettingShareScreenShareOption)shareOption;
/**
 * @brief Gets the option of share application.
 * @return The value is a enumeration of share option.
 */
- (ZoomSDKSettingShareScreenShareOption)getShareOptionWhenShareApplication;
/**
 * @brief Sets the share option when share in meeting.
 * @param shareOption It is a enumeration of share option.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setShareOptionwWhenShareInMeeting:(ZoomSDKSettingShareScreenShareOption)shareOption;
/**
 * @brief Gets the option of share in meeting.
 * @return The value is a enumeration of share option.
 */
- (ZoomSDKSettingShareScreenShareOption)getShareOptionwWhenShareInMeeting;
/**
 * @brief Sets the share option when share in direct share.
 * @param shareOption It is a enumeration of share option.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setShareOptionwWhenShareInDirectShare:(ZoomSDKSettingShareScreenShareOption)shareOption;
/**
 * @brief Gets the option of share in direct share.
 * @return The value is a enumeration of share option.
 */
- (ZoomSDKSettingShareScreenShareOption)getShareOptionwWhenShareInDirectShare;

/**
 * @brief Enable or Disable the GPU acceleration when a user shares video.
 * @param bEnable YES indicates to enable the acceleration, NO indicates to not enable acceleration.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
*/
- (ZoomSDKError)enableHardwareAcceleratedVideoSharing:(BOOL)bEnable;

/**
 * @brief Determines if GPU acceleration is enabled when a user shares video.
 * @return YES if the GPU acceleration is enabled. Otherwise, NO.
 */
- (BOOL)isVideoSharingHardwareAccelerated;

/**
 * @brief Enables or disable Zoom Docs features in custom UI. This is disabled by default.
 * @param enable YES to enable the Zoom Docs feature, NO otherwise.
 * @return If the function succeeds, it returns the ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableZoomDocs:(BOOL)enable;

/**
 * @brief Enable or disable automatic dimming when sharing flashing content.
 * @param enable YES to enable automatic dimming, NO to disable it.
 * @return If the function is success returns @c ZoomSDKError_Success. Otherwise, this function returns an error.
 * @note When enabled, the video automatically dimmed when the shared content contains flashing elements.
*/
-(ZoomSDKError)enableShareContentFlashDetection:(BOOL)enable;

/**
 * @brief Determines if automatic dimming of video when sharing content flashes is enabled.
 * @return YES if the feature is enabled, NO otherwise.
*/
-(BOOL)isShareContentFlashDetectionEnabled;
@end
/**
 * @class ZoomSDKFaceMakeupImageInfo
 * @brief Represents information about a face makeup image in Zoom SDK.
 */
@interface ZoomSDKFaceMakeupImageInfo : NSObject

/**
 * @brief Gets the current image's type.
 */
@property(nonatomic, assign, readonly)ZoomSDKFaceMakeupType faceMakeupType;

/**
 * @brief Determines if the current face makeup item is being used.
 */
@property(nonatomic, assign, readonly)BOOL isSelected;

/**
 * @brief Gets the file path of the current image.
 */
@property(nonatomic, copy, readonly)NSString *imageFilePath;

/**
 * @brief Gets the current image's name.
 */
@property(nonatomic, copy, readonly)NSString *imageName;

/**
 * @brief Gets the current face makeup's index.
 */
@property(nonatomic, assign, readonly)int  index;
@end


/**
 * @protocol ZoomSDKFaceMakeupSettingContextDelegate
 * @brief Delegate protocol to receive callbacks about face makeup setting events.
 */
@protocol ZoomSDKFaceMakeupSettingContextDelegate <NSObject>

/**
 * @brief Callback event notifying that all face makeup items' thumbnails were downloaded.
 * @param type The type of current image.
 */
- (void)onFaceMakeupItemThumbnailsDownloaded:(ZoomSDKFaceMakeupType)type;

/**
 * @brief Callback event notifying that the selected face makeup item is downloading.
 * @param type The type of current image.
 * @param index The current face makeup's index.
 */
- (void)onFaceMakeupItemDataDownloading:(ZoomSDKFaceMakeupType)type index:(int)index;

/**
 * @brief Callback event notifying whether or not the selected face makeup item has downloaded successfully.
 * @param success YES indicates the selected face makeup item has downloaded successfully, NO otherwise.
 * @param type The type of current image.
 * @param index The current face makeup's index.
 */
- (void)onFaceMakeupItemDataDownloaded:(BOOL)success faceMakeupType:(ZoomSDKFaceMakeupType)type index:(int)index;

@end


/**
 * @class ZoomSDKFaceMakeupSettingContext
 * @brief Provides APIs to manage face makeup settings in Zoom SDK.
 */
@interface ZoomSDKFaceMakeupSettingContext : NSObject
{
    id<ZoomSDKFaceMakeupSettingContextDelegate>       _delegate;
}

/**
 * @brief The delegate to receive face makeup setting events.
 */
@property(nonatomic,assign, nullable)id<ZoomSDKFaceMakeupSettingContextDelegate> delegate;

/**
 * @brief Determines if the face makeup feature is enabled.
 * @return YES if face makeup feature is enabled. Otherwise, NO.
 */
- (BOOL)isFaceMakeupEnabled;

/**
 * @brief Determines if the meeting supports the face makeup feature.
 * @return YES if the meeting supports the face makeup feature. Otherwise, NO.
 */
- (BOOL)isSupportFaceMakeup;

/**
 * @brief Enable or Disable the selected face makeup effect is always used by future meetings.
 * @param enable YES indicates enabled, NO otherwise.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)enableFaceMakeupEffectForAllMeeting:(BOOL)enable;

/**
 * @brief Determines if the selected face makeup effect is always used by future meetings.
 * @return YES if the selected face makeup effect still applies to future meetings. Otherwise, NO.
 */
- (BOOL)isFaceMakeupEffectForAllMeetingEnabled;

/**
 * @brief Gets the list of face makeup images.
 * @return If there are images in the list, the pointers's return value is a list to ZoomSDKFaceMakeupImageInfo. Otherwise, this function fails and returns nil.
 */
- (NSArray<ZoomSDKFaceMakeupImageInfo *> *_Nullable)getFaceMakeupImageList;

/**
 * @brief Specify an image to be face makeup image.
 * @param image Specify the image to use.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setFaceMakeupImage:(ZoomSDKFaceMakeupImageInfo *)image;

/**
 * @brief Enable or Disable the lip face makeup effect.
 * @param enable YES indicates enabled, NO otherwise.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setLipsFaceMakeup:(BOOL)enable;

/**
 * @brief Sets the colors of face makeup effect.
 * @param color Specify the face makeup effect's color in RGB format.
 * @param type The specified color uses on which face makeup type.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setColor:(NSColor *)color type:(ZoomSDKFaceMakeupType)type;

/**
 * @brief Sets the transparency of face makeup effect.
 * @param opactity Specify the face makeup effect's transparency. The value should be between 0 to 100.
 * @param type The specified color uses on which face makeup type.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)setOpacity:(int)opacity type:(ZoomSDKFaceMakeupType)type;

/**
 * @brief Disables all the face makeup effect and reset color or opactity value to default value.
 * @return If the function is success returns ZoomSDKError_Success. Otherwise, this function returns an error.
 */
- (ZoomSDKError)resetAllFaceMakeupEffect;
/**
 * @brief Gets the pointer to ZoomSDKSettingTestVideoDeviceHelper for previewing the video with face makeup image.
 * @return If the function succeeds, it returns the pointer to ZoomSDKSettingTestVideoDeviceHelper. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKSettingTestVideoDeviceHelper*_Nullable)getTestVideoDeviceHelper;
@end


/**
 * @class ZoomSDKSettingService
 * @brief Provides access to various Zoom SDK settings modules.
 */
@interface ZoomSDKSettingService : NSObject
{
    ZoomSDKAudioSetting* _audioSetting;
    ZoomSDKVideoSetting* _videoSetting;
    ZoomSDKRecordSetting* _recordSetting;
    ZoomSDKGeneralSetting* _generalSetting;
    ZoomSDKStatisticsSetting* _statisticsSetting;
    ZoomSDKVirtualBackgroundSetting* _virtualBGSetting;
    ZoomSDKShareScreenSetting* _shareScreenSetting;
    ZoomSDK3DAvatarSetting* _avatarSetting;
    ZoomSDKFaceMakeupSettingContext* _faceMakeupSettingContext;
}
/**
 * @brief Gets the object of audio settings.
 * @return If the function succeeds, it returns an object of ZoomSDKAudioSetting. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKAudioSetting*_Nullable)getAudioSetting;

/**
 * @brief Gets the object of video settings.
 * @return If the function succeeds, it returns an object of ZoomSDKVideoSetting. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKVideoSetting*_Nullable)getVideoSetting;

/**
 * @brief Gets the object of meeting recording settings.
 * @return If the function succeeds, it returns an object of ZoomSDKRecordSetting. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKRecordSetting*_Nullable)getRecordSetting;

/**
 * @brief Gets the object of general settings of SDK.
 * @return If the function succeeds, it returns an object of ZoomSDKGeneralSetting. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKGeneralSetting*_Nullable)getGeneralSetting;

/**
 * @brief Gets the object of statistic settings.
 * @return If the function succeeds, it returns an object of ZoomSDKStatisticsSetting. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKStatisticsSetting*_Nullable)getStatisticsSetting;

/**
 * @brief Gets the object of virtual background settings.
 * @return If the function succeeds, it returns an object of ZoomSDKVirtualBackgroundSetting. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKVirtualBackgroundSetting*_Nullable)getVirtualBGSetting;

/**
 * @brief Gets the object of share screen settings.
 * @return If the function succeeds, it returns an object of ZoomSDKShareScreenSetting. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKShareScreenSetting*_Nullable)getShareScreenSetting;

/**
 * @brief Gets the object of 3D avatar.
 * @return If the function succeeds, it returns an object of ZoomSDK3DAvatarSetting. Otherwise, this function fails and returns nil.
 */
- (ZoomSDK3DAvatarSetting*_Nullable)get3DAvatarSettings;

/**
 * @brief Gets the object of face makeup.
 * @return If the function succeeds, it returns an object of ZoomSDKFaceMakeupSettingContext. Otherwise, this function fails and returns nil.
 */
- (ZoomSDKFaceMakeupSettingContext*_Nullable)getFaceMakeupSettings;
/**
 * @brief Custom the url link show or hide on setting page.
 * @param settingPageUrl It is a enumeration of setting page url.
 * @param hide YES to hide, NO to show.
 */
- (void)configToShowUrlLinksInSetting:(ZoomSDKSettingPageURL)settingPageUrl isHide:(BOOL)hide;
@end

NS_ASSUME_NONNULL_END
