#ifndef __ZUI_ENVIRONMENT_H__
#define __ZUI_ENVIRONMENT_H__

#include <map>
#include <memory>
#include <vector>
#include "model/model.h"
#include "view.h"
#include "zui_data_define.h"
#include "window.h"
#include "cmmlib/CmmAppContextSharedDef.h"

namespace zui {

  struct ZUI_API Variant
  {
    enum class DataType
    {
      kInt,
      kFloat,
      kString,
      kFunc,
    };
    DataType type;

    union
    {
      int i;
      float f;
      std::string s;
      std::function<void(const std::string&)> func;
    };

    Variant() : type(DataType::kInt), i() {}
    Variant(int val) : type(DataType::kInt), i(val) {}
    Variant(float val) : type(DataType::kFloat), f(val) {}
    Variant(const std::string& val) : type(DataType::kString) {
      new(&s)std::string(val);
    }
    Variant(const std::function<void(const std::string&)>& f) : type(DataType::kFunc), func(f) {}

    ~Variant()
    {
      switch (type)
      {
      case DataType::kInt:
        break;
      case DataType::kFloat:
        break;
      case DataType::kString:
      {
        using std::string;
        s.std::string::~string();
      }
        break;
      case DataType::kFunc:
        func.~function<void(const std::string&)>();
        break;
      }
    }

    Variant(const Variant& var)
    {
      type = var.type;
      switch (type)
      {
      case DataType::kInt:
        i = var.i;
        break;
      case DataType::kFloat:
        f = var.f;
        break;
      case DataType::kString:
        new(&s)std::string(var.s);
        break;
      case DataType::kFunc:
        func = var.func;
        break;
      }
    }

    Variant& operator=(const Variant& var)
    {
      type = var.type;
      switch (type)
      {
      case DataType::kInt:
        i = var.i;
        break;
      case DataType::kFloat:
        f = var.f;
        break;
      case DataType::kString:
        new(&s)std::string(var.s);
        break;
      case DataType::kFunc:
        new(&func)std::function<void(const std::string&)>(var.func);
        break;
      }
      return *this;
    }
  };

  class ZUI_API Environment
  {
  public:
    static Environment& Instance();

    Environment();
    ~Environment();

    void Insert(const std::string& key, const Variant& val);

    void SetEngine(void* engine) { _engine = engine; }
    void* GetEngine() { return _engine; }

    void SetLanguage(NS_ZOOM_DATA::CMM_LANGUAGE_ID_VALUE languageId);
    NS_ZOOM_DATA::CMM_LANGUAGE_ID_VALUE GetLanguage() const;

    int& GetInt(const std::string& key);
    float& GetFloat(const std::string& key);
    std::string& GetString(const std::string& key);
    std::function<void(const std::string&)>& GetFunc(const std::string& key);

    std::string GetPlatformDelegateTargetId();
    void ScrollToTop(const std::string& scrollViewId);
    
    void OpenUrlWithDefaultBrowser(const std::string& url);
    void OpenSystemNotifications();
    void ScrollToView(const std::string& scrollViewId, const std::string& viewId, const std::string& highlightColor);
    void FocusToView(const std::string& scrollViewId, const std::string& viewId);
    void FocusToView(const std::string& viewId);
    void GetControlTree(const std::string& id, ZUIControlNode& root);

    static void ReleaseUIManager();
    static void ReleaseCustomViews();
  private:
    bool isVail(const std::string& key, Variant::DataType dataType);

  public:
    static const std::string openWindow;
    static const std::string closeWindow;
    static const std::string destoryApp;
    static const std::string openUrl;
    static const std::string clientCMMLanguage;
    static const std::string openSystemNotifications;
    static const std::string language;

  private:
    std::map<std::string, Variant> _data;
    void* _engine = nullptr; // for linux
  };
  static Environment& gEnvironment = Environment::Instance();
}

#endif
