#ifndef __ZUI_EXPRESSION_H__
#define __ZUI_EXPRESSION_H__

namespace zui
{
template<typename T>
class State;

template<typename T>
class Bind;

template <typename Fun, typename... Args>
struct ExpressionTraits {
  using type = typename std::result_of<Fun(Args...)>::type;
};

template <typename Fun, typename... Args>
using ReturnType = typename ExpressionTraits<std::decay_t<Fun>, std::decay_t<Args>...>::type;

template <typename T>
struct IsStateOrBind: std::false_type {
};

template <typename T>
struct IsStateOrBind<State<T>> : std::true_type{};

template <typename T>
struct IsStateOrBind<Bind<T>> : std::true_type {};

template <typename Op, typename L, typename R>
class BinaryOpExpr;

template <typename T>
struct BinaryOpExprTraits : std::false_type {};

template <typename Op, typename L, typename R>
struct BinaryOpExprTraits<BinaryOpExpr<Op, L, R>> : std::true_type {};

template <typename T>
struct IsBinaryOpExpr : BinaryOpExprTraits<T> {};

template <typename L, typename R>
using IsModelOp = std::integral_constant<bool,
  IsStateOrBind<std::decay_t<L>>::value ||
  IsStateOrBind<std::decay_t<R>>::value ||
  IsBinaryOpExpr<std::decay_t<L>>::value ||
  IsBinaryOpExpr<std::decay_t<R>>::value
>;

template <typename T>
struct ValueWrapper {
  using ValueType = std::decay_t<T>;
  T m_value;

  template <typename Type>
  ValueWrapper(Type&& t) : m_value(std::forward<Type>(t)) {}

  const T& operator()() const {
    return m_value;
  }
};

template <typename T>
struct ExprTraitsHelper {
  static constexpr bool cond = IsStateOrBind<T>::value || IsBinaryOpExpr<T>::value;
  using type = typename std::conditional<
    cond,
    T,
    ValueWrapper<T>
  >::type;
};

template <typename T>
using ExprTraits = typename ExprTraitsHelper<T>::type;

template <typename Op, typename L, typename R>
class BinaryOpExpr {
public:
  using ValueType = typename std::decay_t<std::common_type_t<typename L::ValueType, typename R::ValueType>>;

  template <typename Left, typename Right>
  BinaryOpExpr(Left&& l, Right&& r, Op o = Op{})
    : m_left(std::forward<Left>(l)), m_right(std::forward<Right>(r)), m_op(o) {}

  auto operator()() const {
    return calculate();
  }

  operator ValueType() {
    return calculate();
  }

private:
  auto calculate() const {
    return m_op(m_left(), m_right());
  }

  L m_left;
  R m_right;
  Op m_op;
};

template <typename Op, typename L, typename R>
auto make_binary_expr(L&& l, R&& r) {
  return BinaryOpExpr<Op, ExprTraits<std::decay_t<L>>, ExprTraits<std::decay_t<R>>>(
    std::forward<L>(l), std::forward<R>(r));
}

struct AddOp {
  template <typename L, typename R>
  auto operator()(L&& l, R&& r) const -> decltype(std::forward<L>(l) + std::forward<R>(r)) {
    return std::forward<L>(l) + std::forward<R>(r);
  }
};

struct MulOp {
  template <typename L, typename R>
  auto operator()(L&& l, R&& r) const -> decltype(std::forward<L>(l) * std::forward<R>(r)) {
    return std::forward<L>(l) * std::forward<R>(r);
  }
};

struct SubOp {
  template <typename L, typename R>
  auto operator()(L&& l, R&& r) const -> decltype(std::forward<L>(l) - std::forward<R>(r)) {
    return std::forward<L>(l) - std::forward<R>(r);
  }
};

struct DivOp {
  template <typename L, typename R>
  auto operator()(L&& l, R&& r) const -> decltype(std::forward<L>(l) / std::forward<R>(r)) {
    return std::forward<L>(l) / std::forward<R>(r);
  }
};

template <typename T1, typename T2,
  typename = typename std::enable_if<IsModelOp<T1, T2>::value>::type>
  auto operator+(T1&& l, T2&& r)
  -> decltype(make_binary_expr<AddOp>(std::forward<T1>(l), std::forward<T2>(r))) {
  return make_binary_expr<AddOp>(std::forward<T1>(l), std::forward<T2>(r));
}

template <typename T1, typename T2,
  typename = typename std::enable_if<IsModelOp<T1, T2>::value>::type>
  auto operator-(T1&& l, T2&& r)
  -> decltype(make_binary_expr<SubOp>(std::forward<T1>(l), std::forward<T2>(r))) {
  return make_binary_expr<SubOp>(std::forward<T1>(l), std::forward<T2>(r));
}

template <typename T1, typename T2,
  typename = typename std::enable_if<IsModelOp<T1, T2>::value>::type>
  auto operator*(T1&& l, T2&& r)
  -> decltype(make_binary_expr<MulOp>(std::forward<T1>(l), std::forward<T2>(r))) {
  return make_binary_expr<MulOp>(std::forward<T1>(l), std::forward<T2>(r));
}

template <typename T1, typename T2,
  typename = typename std::enable_if<IsModelOp<T1, T2>::value>::type>
  auto operator/(T1&& l, T2&& r)
  -> decltype(make_binary_expr<DivOp>(std::forward<T1>(l), std::forward<T2>(r))) {
  return make_binary_expr<DivOp>(std::forward<T1>(l), std::forward<T2>(r));
}
}
#endif